/*
 * @(#)xabacus.c
 *
 * Copyright 1993 - 2025  David A. Bagley, bagleyd AT verizon.net
 *
 * Abacus demo and neat pointers from
 * Copyright 1991 - 1998  Luis Fernandes, elf AT ee.ryerson.ca
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Driver file for Abacus */
#ifndef WINVER
#include "version.h"
static const char * aboutHelp = {
"Abacus Version " VERSION "\n"
"Send bug reports or fixes to the author: "
"David Bagley <bagleyd AT verizon.net>\n"
"The latest version is at: "
"https://www.sillycycle.com/abacus.html\n"
"Some coding was also done by Luis Fernandes <elf AT ee.ryerson.ca>\n"
"and Sarat Chandran <saratcmahadevan AT yahoo.com>"
};

static const char * synopsisHelp = {
"[-geometry [{width}][x{height}] [{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-{border|bd} {color}] [-frame {color}] [-[no]transparent]\n"
"[-primaryBeadColor {color}] [-leftAuxBeadColor {color}]\n"
"[-rightAuxBeadColor {color}] [-secondaryBeadColor {color}]\n"
"[-highlightBeadColor {color}] [-primaryMarkerColor {color}]\n"
"[-leftAuxMarkerColor {color}] [-rightAuxMarkerColor {color}]\n"
"[-primaryRailColor {color}] [-secondaryRailColor {color}]\n"
"[-highlightRailColor {color}] [-lineRailColor {color}]\n"
"[-bumpSound {filename}] [-moveSound {filename}]\n"
"[-[no]sound] [-delay msecs] [-[no]script] [-[no]demo]\n"
"[-demopath {path}] [-{demofont|demofn} {fontname}]\n"
"[-{demoforeground|demofg} {color}]\n"
"[-[no]teach] [-[no]rightToLeftAdd] [-[no]rightToLeftMult]\n"
"[-[no]lee] [-rails {int}] [-leftAuxRails {int}]\n"
"[-rightAuxRails {int}] [-[no]vertical] [-[no]stippleFrame]\n"
"[-colorScheme {int}] [-[no]slot] [-[no]diamond]\n"
"[-railIndex {int}] [-[no]topOrient] [-[no]bottomOrient]\n"
"[-topNumber {int}] [-bottomNumber {int}] [-topFactor {int}]\n"
"[-bottomFactor {int}] [-topSpaces {int}] [-bottomSpaces {int}]\n"
"[-topPiece {int}] [-bottomPiece {int}] [-topPiecePercent {int}]\n"
"[-bottomPiecePercent {int}] [-shiftPercent {int}]\n"
"[-subdeck {int}] [-subbead {int}] [-[no]sign]\n"
"[-decimalPosition {int}] [-[no]group] [-groupSize {int}]\n"
"[-[no]placeOnRail] -[no]decimalComma] [-base {int}]\n"
"[-subbase {int}] [-[no]subdecksSeparated]\n"
"[-[no]altSubbeadPlacement] [-anomaly {int}] [-shiftAnomaly {int}]\n"
"[-anomalySq {int}] [-shiftAnomalySq {int}] [-displayBase {int}]\n"
"[-[no]pressOffset] [-romanNumerals {none|ancient|modern}]\n"
"[-[no]latin] [-romanMarkers {none|ancient|modern|late|alt}]\n"
"[-{chinese|japanese|korean|russian|danish|roman|medieval|generic}]\n"
"[-museum {it|uk|fr}] [-version]"
};
#endif

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA) || defined(WINVER)
static const char options1Help[] = {
"-geometry {+|-}X{+|-}Y    "
"This option sets the initial position of the abacus "
"window (resource name \"geometry\").\n"
"-display host:dpy    "
"This option specifies the X server to contact.\n"
"-[no]mono       "
"This option allows you to display the abacus window "
"on a color screen as if it were monochrome\n"
"       (resource name \"mono\").\n"
"-[no]{reverse|rv}    "
"This option allows you to see the abacus window in "
"reverse video (resource name\n"
"       \"reverseVideo\").\n"
"-{foreground|fg} color    "
"This option specifies the foreground of the abacus "
"window (resource name \"foreground\").\n"
"-{background|bg} color    "
"This option specifies the background of the abacus "
"window (resource name \"background\").\n"
"-{border|bd} color   "
"This option specifies the color of the border "
"of the beads (resource name \"borderColor\").\n"
"-frame color    "
"This option specifies the color of the frame "
"(resource name \"frameColor\").\n"
"-primaryBeadColor color   "
"This option specifies the color of the beads "
"(resource name \"primaryBeadColor\").\n"
"-leftAuxBeadColor color   "
"This option specifies the color of the beads "
"for the left auxiliary abacus in\n"
"       Lee's Abacus (resource name \"leftAuxBeadColor\").\n"
"-rightAuxBeadColor color  "
"This option specifies the color of the beads "
"for the right auxiliary abacus in\n"
"       Lee's Abacus (resource name \"rightAuxBeadColor\").\n"
"-secondaryBeadColor color "
"This option specifies the secondary color of the beads "
"(resource name\n"
"       \"secondaryBeadColor\").\n"
"-highlightBeadColor color "
"This option specifies the highlight color of the beads "
"(resource name\n"
"       \"highlightBeadColor\").\n"
"-primaryMarkerColor color   "
"This option specifies the foreground of the abacus markings "
"(resource name\n"
"       \"primaryMarkerColor\").\n"
"-leftAuxMarkerColor color   "
"This option specifies the foreground of the abacus markings "
"for the left auxiliary abacus\n"
"       in Lee's Abacus (resource name "
"\"leftAuxMarkerColor\").\n"
"-rightAuxMarkerColor color  "
"This option specifies the foreground of the abacus markings "
"for the right auxiliary abacus\n"
"       in Lee's Abacus (resource name "
"\"rightAuxMarkerColor\").\n"
"-primaryRailColor color   "
"This option specifies the foreground of the rails "
"(resource name \"primaryRailColor\").\n"
"-secondaryRailColor color "
"This option specifies the secondary color of the rails "
"(resource name\n"
"       \"secondaryRailColor\").\n"
"-highlightRailColor color "
"This option specifies the highlight color of the rails "
"(resource name \"highlightRailColor\").\n"
"-lineRailColor color "
"This option specifies the color of the lines when using "
"checkers (resource name \"lineRailColor\").\n"
"-bumpSound filename  "
"This option specifies the file for the bump sound "
"for the movement of the beads.\n"
"-moveSound filename  "
"This option specifies the file for the move sound "
"for the sliding of the decimal point marker.\n"
"-[no]sound      "
"This option specifies if a sliding bead should "
"make a sound or not (resource name \"sound\").\n"
"-delay msecs    "
"This option specifies the number of milliseconds it "
"takes to move a bead or a group of beads\n"
"       one space (resource name \"delay\").\n"
"-[no]script     "
"This option specifies to log application to stdout, "
"every time the user clicks to move the\n"
"       beads (resource name \"script\"). The output is a set "
"of deck, rail, beads added or subtracted, and\n"
"       the number of text lines (4).  This can be edited to add "
"text to the lesson and used as a new demo keeping\n"
"       the generated numbers and the number of lines constant."
"  (Windows version writes to abacus.xml).\n"
"-[no]demo       "
"This option specifies to run in demo mode.  In this "
"mode, the abacus is controlled by the current\n"
"       lesson (resource name \"demo\").  When started with "
"the demo option, a window contains descriptive text,\n"
"       and user prompts are displayed in this window.  Pressing "
"'q' during the demo will quit it.  Clicking the\n"
"       left mouse-button with the pointer in the window "
"will restart the demo (beginning of current lesson).\n"
"       The demo uses abacusDemo.xml and currently there are "
"4 editions possible ((Chinese, Japanese (and\n"
"       Roman), Korean, and Russian (and Danish))."
};
static const char options2Help[] = {
"-demopath path  "
"This option specifies the path for the demo, possibly "
"/usr/local/share/games/xabacus (resource\n"
"       name \"demoPath\"), with the file name of "
"abacusDemo.xml.\n"
"-{demofont|demofn} fontstring    "
"This option specifies the font for the "
"explanatory text that appears in the\n"
"       secondary window, during the demo.  The default font is "
"18 point Times-Roman (-*-times-*-r-*-*-*-180-*).\n"
"       The alternate font is 8x13 (resource name \"demoFont\").\n"
"-{demoforeground|demofg} color   "
"This option specifies the foreground of the abacus "
"demo window (resource name\n"
"       \"demoForeground\").\n"
"-{demobackground|demobg} color   "
"This option specifies the background of the abacus "
"demo window (resource\n"
"       name \"demoBackground\").\n"
"-[no]teach      "
"This option specifies to run in teach mode.  In "
"this mode, the abacus is controlled by 2 numbers\n"
"       separated by an operator: \"+\" for addition, "
"\"-\" for subtraction, \"*\" for multiplication, and \"/\" for\n"
"       division.  The square root operation is represented "
"by the number to be operated on followed by the\n"
"       character \"v\" (this leaves you with an answer "
"from which you must divide by 2).  Similarly, the cube root\n"
"       operation is represented by the number to be operated "
"on followed by the character \"u\" (this leaves you with\n"
"       an answer which from which you must divide by 3).  "
"Press return key to progress through the steps\n"
"       (resource name \"teach\").\n"
"-[no]rightToLeftAdd  "
"This option specifies the order for teach starting "
"side for addition and subtraction.  The\n"
"       default is the traditional left to right.  "
"Right to left seems easier though (resource name \"rightToLeftAdd\").\n"
"-[no]rightToLeftMult "
"This option specifies the order for teach starting "
"side for multiplication.  The default is\n"
"       the traditional left to right.  Right to left "
"seems more straight forward (resource name \"rightToLeftMult\").\n"
"-[no]lee   "
"This option allows you to turn on and off the two extra "
"auxiliary abaci (resource name \"lee\").\n"
"-rails int      "
"This option specifies the number of rails (resource "
"name \"rails\").\n"
"-leftAuxRails int "
"This option allows you to set the number of the "
"rails for the left auxiliary abacus in Lee's Abacus\n"
"       (resource name \"leftAuxRails\").\n"
"-rightAuxRails int  "
"This option allows you to set the number of the "
"rails for the right auxiliary abacus in Lee's\n"
"       Abacus (resource name \"rightAuxRails\").\n"
"-[no]vertical "
"This option allows you to have vertical rails "
"(resource name \"vertical\").\n"
"-[no]stippleFrame "
"This option allows you to toggle the frame stippling "
"(resource name \"stippleFrame\").\n"
"-colorScheme int    "
"This option specifies the color scheme for the "
"abacus (resource name \"colorScheme\") where\n"
"       0-> none, 1-> darken middle bead (2 beads "
"if even), 2-> darken first bead of a group,\n"
"       4-> use secondary color for second half of a "
"row of beads (but if odd color middle bead),\n"
"       8-> use secondary color in alternate groups."
"  Use a mask of 15 for combinations.\n"
"-[no]slot       "
"This option allows you to have either slots or rails "
"(resource name \"slot\").\n"
"-[no]diamond    "
"This option allows you to have diamond or round beads "
"(resource name \"diamond\").\n"
"-railIndex int  "
"This option specifies the index of color for the "
"rails of the abacus (resource name \"railIndex\")\n"
"       where a value is 0 or 1."
};
static const char options3Help[] = {
"-[no]topOrient    "
"This option specifies the orientation of the beads "
"on top (resource name \"topOrient\").\n"
"-[no]bottomOrient    "
"This option specifies the orientation of the beads "
"on bottom (resource name \"bottomOrient\").\n"
"-topNumber int    "
"This option specifies the number of beads on top "
"(resource name \"topNumber\").\n"
"-bottomNumber int    "
"This option specifies the number of beads on bottom "
"(resource name \"bottomNumber\").\n"
"-topFactor int    "
"This option specifies the multiply factor for the "
"beads on top (resource name \"topFactor\").\n"
"-bottomFactor int    "
"This option specifies the multiply factor for the "
"beads on bottom (resource name\n"
"       \"bottomFactor\").\n"
"-topSpaces int    "
"This option specifies the number of spaces on top "
"(resource name \"topSpaces\").\n"
"-bottomSpaces int    "
"This option specifies the number of spaces on bottom "
"(resource name \"bottomSpaces\").\n"
"-topPiece int     "
"This option specifies the number of pieces on top "
"(resource name \"topPiece\").\n"
"-bottomPiece int     "
"This option specifies the number of pieces on bottom "
"(resource name \"bottomPiece\").\n"
"-topPiecePercent int   "
"This option specifies the number of piece percents on top "
"(resource name\n"
"       \"topPiecePercent\").\n"
"-bottomPiecePercent int   "
"This option specifies the number of piece percents on bottom "
"(resource name\n"
"       \"bottomPiecePercent\").\n"
"-shiftPercent int    "
"This option specifies the shift of rails for piece percents "
"and also may influence the\n"
"       precision of the calculation (resource name "
"\"shiftPercent\").\n"
"-subdeck int    "
"This option specifies the special subdecks column "
"(resource name " "\"subdeck\").\n"
"-subbead int    "
"This option specifies the special subbeads column "
"(resource name " "\"subbead\").\n"
"-[no]sign       "
"This option allows you to set the abacus to allow "
"negatives (resource name \"sign\").\n"
"-decimalPosition int "
"This option specifies the number of rails to the "
"right of the decimal point\n"
"       (normally 2) (resource name \"decimalPosition\").\n"
"-[no]group      "
"This option allows you to group the displayed "
"digits for readability (resource name \"group\").\n"
"-groupSize int       "
"This option specifies the group size to the left of the "
"decimal point (normally 3) (resource\n"
"       name \"groupSize\").\n"
"-[no]placeOnRail     "
"This option allows you to place the decimal and group "
"separators on rail (or just after)\n"
"       (resource name \"placeOnRail\").\n"
"-[no]decimalComma    "
"This option allows you to swap \".\" for \",\" "
"to allow for different display format (resource\n"
"       name \"decimalComma\").\n"
"-base int       "
"This option specifies the base used (default is base "
"10) (resource name \"base\").\n"
"-subbase int    "
"This option specifies the base for the Roman subdeck, "
"can set to 8 or default of 12 (resource\n"
"name \"subbase\").\n"
"-[no]subdecksSeparated "
"This option allows you to have the subdecks separated "
"(resource name \"subdecksSeparated\").\n"
"-[no]altSubbeadPlacement "
"This option allows you to have the subdeck symbols on "
"the left side (resource " "name\n"
"\"altSubbeadPlacement\").\n"
"-anomaly int         "
"This option specifies the offset from base for a "
"multiplicative factor of the rail with the\n"
"       anomaly (if none, this is set to 0) "
"(resource name \"anomalySq\").\n"
"-shiftAnomaly int    "
"This option specifies the offset from decimal point "
"for the anomaly (usually 2) (resource name\n"
"       \"shiftAnomaly\").\n"
"-anomalySq int       "
"This option specifies the offset from base for the "
"second anomaly (if none, this is set to 0)\n"
"       (resource name \"anomalySq\").\n"
"-shiftAnomalySq int  "
"This option specifies the offset in rails from the "
"first anomaly (usually 2) (resource\n"
"       name \"shiftAnomalySq\")."
};
static const char options4Help[] = {
"-displayBase int     "
"This option specifies the base displayed (default is "
"base 10) (resource name \"displayBase\").\n"
"       If this is different then \"base\" then it is "
"implemented using \"long long\" and the calculation is\n"
"       limited by its bounds.  Also the fractional "
"part does not scale with the \"displayBase\" so if the\n"
"       \"displayBase\" is greater than the \"base\" it "
"looses some precision.  Also no rounding is done.\n"
"-[no]pressOffset     "
"This option allows you to put a pixel space between all the "
"beads so there is room for the bead\n"
"       to move when pressed (resource name \"pressOffset\").\n"
"-romanNumerals   {none|ancient|modern} "
"This option allows you to set the abacus to "
"allow Roman Numerals (resource name\n"
"       \"romanNumerals\").  Modern Roman Numerals above 3999 "
"are normally represented with bars on top, due to ASCII\n"
"       constraints this is represented instead in lower case "
"(historically case was ignored).   Roman Numerals above\n"
"       3,999,999 were not represented historically.  "
"Roman numerals change with displayBase in an\n"
"       \"experimental\" way.  When used with twelfths and "
"subdecks, named fraction symbols are used.  Due to\n"
"       ASCII constraints the sigma is represented as E, the "
"backwards C is represented as a Q, the mu as a u, and\n"
"       the Z with a - through the center as a z.  If "
"available, decimal input is ignored.\n"
"-[no]latin  "
"This option allows you to set the abacus to "
"allow latin fractions instead of a symbolic notation in the\n"
"       Roman numeral output (resource name "
"\"latin\").\n"
"-romanMarkers     {none|ancient|modern|late|alt} "
"This option allows you to set Roman numerals of"
"a specific type on the\n"
"       frame of the abacus (resource name \"romanMarkers\").\n"
"-chinese    "
"This option specifies the format of the abacus "
"(resource name \"format\") to \"Chinese\" for the Chinese\n"
"       Suanpan.\n"
"-japanese   "
"This option specifies the format of the abacus "
"(resource name \"format\") to \"Japanese\" for the\n"
"       Japanese post-WWII Soroban.  "
"This is also similar to the Roman Hand Abacus.\n"
"-korean     "
"This option specifies the format of the abacus "
"(resource name \"format\") to \"Korean\" for the Korean\n"
"       Jupan or Japanese pre-WWII Soroban.\n"
"-russian    "
"This option specifies the format of the abacus "
"(resource name \"format\") to \"Russian\" for the Russian\n"
"       Schoty.  To complete, specify \"piece\" resource to be 4.\n"
"-danish     "
"This option specifies the format of the abacus "
"(resource name \"format\") to \"Danish\" for the Danish\n"
"       Elementary School Abacus teaching aid.\n"
"-roman      "
"This option specifies the format of the abacus "
"(resource name \"format\") to \"Roman\" for the Roman\n"
"       Hand Abacus, note beads will move in slots.  "
"To complete, specify \"romanNumerals\".\n"
"-medieval   "
"This option specifies the format of the abacus "
"(resource name \"format\") to \"Medieval\" for the Medieval\n"
"       Counter, with counters instead of beads.\n"
"-generic    "
"This option specifies the format of the abacus "
"(resource name \"format\") to \"Generic\".  This option\n"
"       specifies a format that is more configurable by using "
"using resources, since there are few rules to govern its\n"
"       behavior.\n"
"-museum         {it|br|fr} "
"This option specifies the country code of the museum of "
"the abacus in Museum of the Thermae,\n"
"       Rome, the British Museum, London, or the Cabinet de "
"medailles, Bibliotheque nationale, Paris.\n"
"-version    "
"This option tells you what version of xabacus you have."
};

static const char description1Help[] = {
"This is an implementation of the classic Chinese Abacus "
"(Suanpan) which has its origins in the 12th century.\n"
"\n"
"The device has two decks.  Each deck, separated by a "
"partition, normally has 13 rails on which are mounted beads.\n"
"Each rail on the top deck contains 1 or 2  beads, and each "
"rod on the bottom deck contains 4 or 5 beads.  Each bead on\n"
"the upper deck has a value of five, while each bead on the "
"lower deck has value of one.  Beads are considered counted,\n"
"when moved towards the partition separating the decks, i.e. "
"to add a value of one, a bead in the bottom deck is moved\n"
"up, and to add a value of 5, a bead in the top deck is moved "
"down.\n"
"\n"
"The basic operations of the abacus are addition and subtraction. "
" Multiplication can be done by mentally multiplying\n"
"the digits and adding up the intermediate results on the abacus. "
" Division would be similar where the intermediate\n"
"results are subtracted.  There are techniques like using your "
"thumb with forefinger which does not apply with mouse\n"
"entry.  Also with multiplication, one can carry out "
"calculations on different parts of the abacus for scratch work,\n"
"here it is nice to have a long abacus.\n"
"\n"
"The pre-WWII Japanese Abacus (Soroban) (or Korean Jupan) "
"is similar to the Chinese Abacus but has only one bead\n"
"per rail on the top deck.  The later Japanese Abacus was"
"further simplified to have only 4 beads per rail on the\n"
"bottom deck.\n"
"\n"
"The Russian Abacus was invented in the 17th century, here "
"the beads are moved from right to left.  It has colored\n"
"beads in the middle for ease of use.  Quarters represent "
"1/4 Rubles and are only  present historically on the Russian\n"
"Abacus (Schoty).  Some of the older Schoty have a extra place "
"for the 1/4 Kopek (quarter percent) as well as the 1/4\n"
"Ruble (quarter)."
};

static const char description2Help[] = {
"\n"
"The Danish Abacus was used in the early 20th century in "
"elementary schools as a teaching aid.\n"
"\n"
"The Roman Hand-Abacus predates the Chinese Abacus and is "
"very similar to the later Japanese Abacus, but seems to\n"
"have fallen out of use with the Fall of the Roman Empire "
"(at least 3 are in existence).  The Roman Abaci are brass\n"
"plates where the beads move in slots.  In addition to the "
"normal 7 columns of beads, they generally have 2 special\n"
"columns on the right side.  In two examples: the first "
"special column was for 12ths (12 uncia (ounces) = 1 as) and\n"
"had one extra bead in the bottom deck.  Also the last column "
"was a combination of halves, quarters, and twelfths of an\n"
"ounce and had no beads in the top deck and 4 beads at the "
"bottom (beads did not have to come to the top to be\n"
"counted but at one of 3 marked points, where the top bead "
"was for halves, the next bead for quarters, and the last two\n"
"beads for twelfths).  In another surviving example: the 2 "
"special columns were switched and the combination column\n"
"was broken into 3 separate slots.  If available, decimal "
"input is ignored.\n"
"\n"
"The Medieval Counter is a primitive form of the abacus and "
"was used in Europe as late as the 1600s.  It was useful\n"
"considering they were using it with Roman Numerals.   This "
"is similar to the Salamis Greek Tablet from 4th or 5th\n"
"Century BCE.\n"
"\n"
"The Mesoamerican Nepohualtzintzin is a Japanese Abacus "
"base 20.  The Mesoamericans had base 20 with the\n"
"exception of the 3rd decimal place where instead of "
"20 * 20 = 400 the third place marked 360 and the 4th place was\n"
"20 * 360, etc.  They independently created their own zero "
"(only the Sumerian and early Babylonian culture (base 60)\n"
"and India (base 10) have done this) but the anomaly took "
"away its true power.\n"
"\n"
"An easy way of figuring out time in seconds given hours, "
"minutes, and seconds, can be done on the abacus with\n"
"Sumerian abacus with topFactor 10, topNumber 5 and "
"bottomNumber 9 using base 60.  No written record of such a\n"
"device exits but its plausible that one did.\n"
"\n"
"The Chinese Solid-and-Broken-Bar System is a base 12 "
"numbering system and not really an abacus.  When the\n"
"abacus is setup in this way though (topFactor 3, topNumber 3, "
"bottomNumber 2, base 12, displayBase 12), it is easy to\n"
"relate the two.\n"
"\n"
"The signed bead is an invention of the author and is not "
"present on any historical abacus (to his knowledge) and is\n"
"used to represent negatives.  \"New & Improved\" abacus "
"models have two auxiliary decks stacked above the principal\n"
"deck that enable multiplication, division, square-root, and "
"cube-root computations to be performed with equal ease as\n"
"addition and subtraction (well, so I have read)."
};

static const char featuresHelp[] = {
"Click \"mouse-left\" button on a bead you want to move.  The "
"beads will shift themselves to vacate the area of the\n"
"column that was clicked.\n"
"\n"
"Click \"mouse-right\" button, or press \"C\" or \"c\" keys, "
"to clear the abacus.\n"
"\n"
"Press \"-\" or \"_\" keys to complement the beads on the rails.\n"
"\n"
"Press \"U\" or \"u\" keys to undo last move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo the last undo.\n"
"\n"
"Press \"I\" or \"i\" keys to increment the number of rails.\n"
"Press \"D\" or \"d\" keys to decrement the number of rails.\n"
"\n"
"Press \"F\" or \"f\" keys to switch between Chinese, "
"Japanese, Korean, Russian, Danish, Roman, and Medieval formats.\n"
"There is an extra \"Generic\" format, this allows one to "
"break some rules binding the other formats (for example, if\n"
"one wanted more beads on top deck than on bottom deck you "
"would use this, in addition to resource option changes).\n"
"\n"
"Press \"V\" or \"v\" keys to change the Roman Nvmerals "
"format.  (Pardon humor/typo but ran out of letters).\n"
"\n"
"Press \"~\" key to toggle Latin Numerals (when Roman "
"Nvmerals and quarter beads or twelfth beads are activated).\n"
"\n"
"Press \"G\" or \"g\" keys to toggle (usu.) commas between "
"groups of digits.\n"
"\n"
"Press \"#\" key to toggle place on rail (or between rails) "
"for group and decimal separators.\n"
"\n"
"Press \"|\" key to toggle vertical placement (best for "
"Russian and Danish Abacus, and Medieval Counters).\n"
"\n"
"Press \"X\" or \"x\" keys to toggle the frame stippling.\n"
"\n"
"Press \"S\" or \"s\" keys to toggle the sign bead.\n"
"\n"
"Press \"P\" or \"p\" keys to change the piece setting "
"for none|quarter|eighth|twelfth beads.  Originally\n"
"intended for the Roman Hand Abacus (twelfths) and the "
"Russian Abacus (quarter rubles).\n"
"\n"
"Press \"T\" or \"t\" keys to change the piece percent "
"setting for none|quarter|eighth|twelfth beads.  Originally\n"
"intended for the old Russian Abacus (quarter kopeks).\n"
"\n"
"Press \"B\" or \"b\" keys to change the subdeck setting "
"to be interpreted as none|quarter|eighth|twelfth beads.\n"
"Originally intended for the Roman Hand Abacus (twelfths "
"and possibly eighths).  For twelfths, the lowest value\n"
"of two at bottom of the rightmost column of beads are "
"a twelfth of the column second from right.  Middle is\n"
"worth a quarter and the top is worth a half.  For "
"eighths, middle and bottom are both worth an eight and\n"
"the top a half.  (For quarters, all are worth a quarter.)\n"
"\n"
"Press \"K\" or \"k\" to toggle the subdeck separation.\n"
"\n"
"Press \"L\" or \"l\" to toggle which side the subdeck "
"symbols are placed.\n"
"\n"
"Press \"M\" or \"m\" keys to switch between it, uk, and fr "
"museum formats.\n"
"\n"
"Press \"Z\" or \"z\" keys to change the Roman markerz "
"on the frame.\n"
"\n"
"Press \"Y\" or \"y\" keys to toggle the availability of "
"anomaly bars.   Intended to used with Japanese Abacus and\n"
"base 20 for the Mesoamerican Abacus. (Mutually exclusive "
"to watch bars).\n"
"\n"
"Press \"W\" or \"w\" keys to toggle the availability of "
"watch bars.  Intended to represent seconds where hours\n"
"and minutes can be set.  (Mutually exclusive to anomaly "
"bars).\n"
"\n"
"Press \"O\" or \"o\" keys to toggle the demo mode.\n"
"\n"
"Press \"$\" key to toggle the teach mode.\n"
"\n"
"In teach mode, \"+\" key toggles starting side to sum, "
"\"*\" key toggles for starting side for multiplicand.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of beads.  "
"Press \"<\" or \",\" keys to slow down this movement.\n"
"\n"
"Press \"@\" key to toggle the sound.  "
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"The abacus may be resized.  Beads will reshape depending "
"on the room they have.\n"
"\n"
"Demo Mode: in this mode, the abacus is controlled by the "
"program.  When started with the demo option, a second\n"
"window is presented that should be placed directly below the "
"abacus-window. Descriptive text, and user prompts are\n"
"displayed in this window.  Pressing 'q' during the demo "
"will quit it.  Clicking the left mouse-button with the\n"
"pointer in the window will restart the demo (beginning "
"of current lesson)."
};
static const char referencesHelp[] = {
"Luis Fernandes  http://www.ee.ryerson.ca/~elf/abacus/\n"
"Lee Kai-chen, How to Learn Lee's Abacus, 1958, 58 pages.\n"
"Abacus Guide Book, 57 pages.\n"
"Georges Ifrah, The Universal history of Numbers, Wiley Press "
"2000, pp 209-211, 288-294.\n"
"Review of above: http://www.ams.org/notices/200201/rev-dauben.pdf\n"
"David Eugene Smith, History of Mathematics Volume II, "
"Dover Publications, Inc 1958, pp 156-195."
};

#define DESCRIPTION1_SIZE (sizeof (description1Help) + 1)
#define DESCRIPTION2_SIZE (sizeof (description2Help) + 1)
#define DESCRIPTION_SIZE (DESCRIPTION1_SIZE + DESCRIPTION2_SIZE)
char descriptionHelp[DESCRIPTION_SIZE];
#define OPTIONS1_SIZE (sizeof (options1Help) + 1)
#define OPTIONS2_SIZE (sizeof (options2Help) + 1)
#define OPTIONS3_SIZE (sizeof (options3Help) + 1)
#define OPTIONS4_SIZE (sizeof (options4Help) + 1)
#define OPTIONS_SIZE (OPTIONS1_SIZE + OPTIONS2_SIZE + OPTIONS3_SIZE + OPTIONS4_SIZE)
char optionsHelp[OPTIONS_SIZE];
#endif

static const char *formatChoices[] =
{
	"Chinese Suanpan",
	"Japanese Soroban",
	"Korean Jupan",
	"Russian Schoty",
	"Danish School Abacus",
	"Roman Hand-abacus",
	"Medieval Counter",
	"Generic Abacus"
};

#define PIECE_CHOICE(t, b) (((t == 0) ? b : (t * b)) >> 2)
#define PIECE_BASE(i) (i << 2)
#define PIECE_TOP(i) ((i > 1) ? 2 : 0)
#define PIECE_BOTTOM(i) ((value > 2) ? 6 : ((value == 0) ? 0 : 4))
#ifdef ANOMALY
#define ANOMALY_CHOICE(a, as) ((a == 2 && as == 0) ? 1 : ((a == 4 && as == 4) ? 2 : 0))
#define ANOMALY_P(i) (i << 1)
#define ANOMALY_SQ(i) ((i == 2) ? 4 : 0)
#endif

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static const char *romanFormatChoices[] =
{
	"None",
	"Ancient",
	"Modern",
	"Late",
	"Alt"
};

static const char *pieceChoices[] =
{
	"None",
	"Quarter",
	"Eighth",
	"Twelfth"
};

static const char *museumChoices[] =
{
	"Italian",
	"British",
	"French"
};

#ifdef ANOMALY
static const char *anomalyChoices[] =
{
	"None",
	"Calendar",
	"Watch"
};
#endif
#endif

#define ENABLE_LATIN(romanNumeralsMode, bottomPiece) \
  (romanNumeralsMode != NONE && bottomPiece != 0)
#define ENABLE_PIECE_PERCENT(bottomPiece, bottomPiecePercent, decimalPosition) \
  (((bottomPiecePercent != 0) ? 1 : 0) + decimalPosition >= \
  2 + ((bottomPiece != 0) ? 1 : 0))
#define ENABLE_SECONDARY(bottomPiece, bottomPiecePercent, decimalPosition, slot) \
  (bottomPiece != 0 && bottomPiecePercent == 0 && decimalPosition == 3 && slot)
#define ENABLE_SUBDECKS(bottomPiece, bottomPiecePercent, decimalPosition, slot, subbase) \
  (ENABLE_SECONDARY(bottomPiece, bottomPiecePercent, decimalPosition, slot) && subbase > 0)
#define ENABLE_MUSEUM(bottomPiece, bottomPiecePercent, decimalPosition, slot, subbase, romanMarkersMode, anomaly, anomalySq) \
  (ENABLE_SECONDARY(bottomPiece, bottomPiecePercent, decimalPosition, slot) && subbase > 0 && romanMarkersMode > 0 && anomaly == 0 && anomalySq == 0)
#define ENABLE_ROMAN_MARKERS(anomaly, anomalySq) (anomaly == 0 && anomalySq == 0)

#define CALC_STRING_SIZE 120

#ifdef WINVER
#include "AbacusP.h"
#define TITLE "wabacus"

static AbacusRec widget;

char calc[CALC_STRING_SIZE] = "";

#define MAX_DIGITS 256		/* This limits the number of rails */
#define MAX_DIGITS_DISPLAY (MAX_DIGITS+67)

static void
forceNormalParams(AbacusWidget w)
{
	int base = DEFAULT_BASE;
	int displayBase = DEFAULT_BASE;

	w->abacus.base = base;
	w->abacus.displayBase = displayBase;
}

static void
forceDemoParams(AbacusWidget w)
{
	int min;
	unsigned int mode;
	int bottomPiece, bottomPiecePercent;
	Boolean demo, sign;

	forceNormalParams(w);
	mode = w->abacus.mode;
	sign = w->abacus.sign;
	bottomPiece = w->abacus.decks[BOTTOM].piece;
	bottomPiecePercent = w->abacus.decks[BOTTOM].piecePercent;
	demo = w->abacus.demo;
	if (mode == GENERIC) {
		changeFormatAbacus(w);
	}
	min = ((sign) ? 1: 0) + ((bottomPiece) ? 1 : 0) +
		((bottomPiecePercent) ? 1 + w->abacus.shiftPercent : 0) +
		((demo) ? MIN_DEMO_RAILS : MIN_RAILS);
	while (w->abacus.rails < min) {
		incrementAbacus(w);
	}
}

static void
forceTeachParams(AbacusWidget w)
{
#ifndef BASE_TEACH
	w->abacus.displayBase = w->abacus.base;
#endif
	w->abacus.sign = False;
	w->abacus.anomaly = 0;
	w->abacus.anomalySq = 0;
	/*if (w->abacus.mode == GENERIC) {
		changeFormatAbacus(w);
	}*/
	(void) InvalidateRect(w->core.hWnd, NULL, TRUE);
}

static void
checkEnabled(AbacusWidget w)
{
	EnableMenuItem(w->core.hMenu, ACTION_LATIN,
		(unsigned) (ENABLE_LATIN(w->abacus.romanNumeralsMode, w->abacus.decks[BOTTOM].piece) ?
		MF_ENABLED : MF_GRAYED));
	EnableMenuItem(w->core.hMenu, ACTION_PIECE_PERCENT,
		(unsigned) (ENABLE_PIECE_PERCENT(w->abacus.decks[BOTTOM].piece, w->abacus.decks[BOTTOM].piecePercent, w->abacus.decimalPosition) ?
		MF_ENABLED : MF_GRAYED));
	EnableMenuItem(w->core.hMenu, ACTION_SUBDECK,
		(unsigned) (ENABLE_SECONDARY(w->abacus.decks[BOTTOM].piece, w->abacus.decks[BOTTOM].piecePercent, w->abacus.decimalPosition, w->abacus.slot) ?
		MF_ENABLED : MF_GRAYED));
	EnableMenuItem(w->core.hMenu, ACTION_SUBDECKS_SEPARATED,
		(unsigned) (ENABLE_SUBDECKS(w->abacus.decks[BOTTOM].piece, w->abacus.decks[BOTTOM].piecePercent, w->abacus.decimalPosition, w->abacus.slot, w->abacus.subbase) ?
		MF_ENABLED : MF_GRAYED));
	EnableMenuItem(w->core.hMenu, ACTION_ALT_SUBBEAD_PLACEMENT,
		(unsigned) (ENABLE_SUBDECKS(w->abacus.decks[BOTTOM].piece, w->abacus.decks[BOTTOM].piecePercent, w->abacus.decimalPosition, w->abacus.slot, w->abacus.subbase) ?
		MF_ENABLED : MF_GRAYED));
	EnableMenuItem(w->core.hMenu, ACTION_MUSEUM,
		(unsigned) (ENABLE_MUSEUM(w->abacus.decks[BOTTOM].piece, w->abacus.decks[BOTTOM].piecePercent, w->abacus.decimalPosition, w->abacus.slot, w->abacus.subbase, w->abacus.romanMarkersMode, w->abacus.anomaly, w->abacus.anomalySq) ?
		MF_ENABLED : MF_GRAYED));
	EnableMenuItem(w->core.hMenu, ACTION_ROMAN_MARKERS,
		(unsigned) (ENABLE_ROMAN_MARKERS(w->abacus.anomaly, w->abacus.anomalySq) ?
		MF_ENABLED : MF_GRAYED));
}

void
setAbacus(AbacusWidget w, int reason)
{
	switch (reason) {
	case ACTION_SCRIPT:
#if 0
		{
			int deck, rail, number;

			deck = w->abacus.deck;
			rail = w->abacus.rail;
			number = w->abacus.number;
			(void) printf("%d %d %d %d\n", PRIMARY,
				deck, rail, number);
		}
#endif
		break;
	case ACTION_BASE_DEFAULT:
		forceNormalParams(w);
		break;
	case ACTION_DEMO_DEFAULT:
		clearAbacus(w);
		forceDemoParams(w);
		break;
	case ACTION_CLEAR:
		clearAbacus(w);
		if (w->abacus.demo) {
			clearAbacusDemo(w);
		}
		break;
	case ACTION_DECIMAL_CLEAR:
		clearDecimalAbacus(w);
		break;
	case ACTION_UNDO:
		undoAbacus(w);
		break;
	case ACTION_REDO:
		redoAbacus(w);
		break;
	case ACTION_COMPLEMENT:
		complementAbacus(w);
		break;
	case ACTION_CLEAR_NODEMO:
		clearAbacus(w);
		break;
	case ACTION_HIDE:
		ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
		break;
	case ACTION_CLEAR_QUERY:
	case ACTION_PAUSE_QUERY:
		break;
	case ACTION_DEMO:
		toggleDemoAbacusDemo(w);
		break;
	case ACTION_NEXT:
		showNextAbacusDemo(w);
		break;
	case ACTION_ENTER:
		showEnterAbacusDemo(w);
		break;
	case ACTION_JUMP:
		showJumpAbacusDemo(w);
		break;
	case ACTION_MORE:
		showMoreAbacusDemo(w);
		break;
	case ACTION_INCREMENT:
	case ACTION_DECREMENT:
	case ACTION_FORMAT:
	case ACTION_ROMAN_NUMERALS:
	case ACTION_ROMAN_MARKERS:
	case ACTION_LATIN:
	case ACTION_GROUP:
	case ACTION_PLACE_ON_RAIL:
	case ACTION_VERTICAL:
	case ACTION_STIPPLE_FRAME:
	case ACTION_SIGN:
	case ACTION_PIECE:
	case ACTION_PIECE_PERCENT:
	case ACTION_SUBDECK:
	case ACTION_MUSEUM:
	case ACTION_SUBDECKS_SEPARATED:
	case ACTION_ALT_SUBBEAD_PLACEMENT:
#ifdef ANOMALY
	case ACTION_CALENDAR_ANOMALY:
	case ACTION_WATCH_ANOMALY:
#endif
		break;
	}
}

/* input dialog box handle */
HWND hCalcDlg = NULL;
HWND hDemoDlg = NULL;
HWND hTeachDlg = NULL;

void
setAbacusString(AbacusWidget w, int reason, char *string)
{
	(void) strncpy(calc, string, CALC_STRING_SIZE - 1);
	calc[CALC_STRING_SIZE - 1] = '\0';
	if (reason == ACTION_SCRIPT || reason == ACTION_IGNORE) {
		char szBuf[MAX_DIGITS_DISPLAY];
		unsigned int mode = w->abacus.mode;

#ifdef HAVE_SNPRINTF
		(void) snprintf(szBuf, MAX_DIGITS_DISPLAY,
			"%s  %s", string,
			(mode >= MAX_FORMATS) ? "Abacus" :
			formatChoices[mode]);
#else
		(void) sprintf(szBuf,
			"%s  %s", string,
			(mode >= MAX_FORMATS) ? "Abacus" :
			formatChoices[mode]);
#endif
		SetWindowText(w->core.hWnd, (LPSTR) szBuf);
	}
	SetDlgItemText(hCalcDlg, ACTION_CALC, calc);
}

static LRESULT CALLBACK
about(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND &&
			(LOWORD(wParam) == IDOK ||
			LOWORD(wParam) == IDCANCEL)) {
		(void) EndDialog(hDlg, TRUE);
		return TRUE;
	}
	return FALSE;
}

static LRESULT CALLBACK
teach(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND) {
		if (LOWORD(wParam) == IDOK) {
			char str[CALC_STRING_SIZE];
			GetDlgItemText(hDlg, ACTION_TEACH, str, CALC_STRING_SIZE);
			widget.core.hDC = GetDC(widget.core.hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			widget.abacus.teach = True;
			(void) teachStep(&widget, str, 0);
			if (strcmp(str, "q") == 0) {
				destroyAbacus(&widget, NULL);
			}
			/*setAbacusString(&widget, ACTION_IGNORE, str);*/
			(void) ReleaseDC(widget.core.hWnd,
				widget.core.hDC);
			/*SetDlgItemText(hDlg, ACTION_TEACH, calc);*/
			return TRUE;
		} else if (LOWORD(wParam) == IDCANCEL) {
			widget.abacus.teach = False;
			(void) EndDialog(hDlg, TRUE);
			hTeachDlg = NULL;
			(void) InvalidateRect(widget.core.hWnd, NULL, TRUE);
			return TRUE;
		}
	}
	return FALSE;
}

static LRESULT CALLBACK
calculation(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND) {
		if (LOWORD(wParam) == IDOK) {
			int i, j = 0;
			char mathBuff[CALC_STRING_SIZE];
			GetDlgItemText(hDlg, ACTION_CALC, mathBuff, CALC_STRING_SIZE);
			widget.core.hWnd = GetParent(hDlg);
			widget.core.hDC = GetDC(widget.core.hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			/* strip out blanks */
			for (i = 0; i < (int) strlen(mathBuff); i++) {
				if (mathBuff[i] == '[' || mathBuff[i] == ']') {
					mathBuff[j] = '\0';
					break;
				} else if (mathBuff[i] != ' ' &&
						mathBuff[i] != '\t') {
					mathBuff[j] = mathBuff[i];
					j++;
				}
			}
			calculate(&widget, mathBuff, 0);
			(void) ReleaseDC(widget.core.hWnd,
				widget.core.hDC);
			SetDlgItemText(hDlg, ACTION_CALC, calc);
			return TRUE;
		} else if (LOWORD(wParam) == IDCANCEL) {
			(void) DestroyWindow(hDlg);
			hCalcDlg = NULL;
			return TRUE;
		}
	}
	return FALSE;
}

static LRESULT CALLBACK
demo(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND) {
		if (LOWORD(wParam) == IDOK) {
			char str[CALC_STRING_SIZE];
			GetDlgItemText(hDlg, ACTION_DEMO1, str, CALC_STRING_SIZE);
			widget.core.hWnd = GetParent(hDlg);
			widget.core.hDC = GetDC(widget.core.hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			/*demo(&widget, str, 0);*/
			(void) ReleaseDC(widget.core.hWnd,
				widget.core.hDC);
			SetDlgItemText(hDlg, ACTION_DEMO4, "fix me");
			return TRUE;
		} else if (LOWORD(wParam) == IDCANCEL) {
			widget.abacus.demo = False;
			(void) EndDialog(hDlg, TRUE);
			hDemoDlg = NULL;
			(void) InvalidateRect(widget.core.hWnd, NULL, TRUE);
			return TRUE;
		}
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.abacus.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			enterAbacus(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.abacus.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			leaveAbacus(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
	case WM_CREATE:
		initializeAbacus(&widget, brush);
		widget.core.hMenu = GetMenu(widget.core.hWnd);
		checkEnabled(&widget);
		initializeAbacusDemo(&widget);
		break;
	case WM_DESTROY:
		destroyAbacus(&widget, brush);
#if 0
		if (widget.abacus.demo)
			destroyAbacusDemo();
#endif
		break;
	case WM_SIZE:
		resizeAbacus(&widget);
		(void) InvalidateRect(hWnd, NULL, TRUE);
		break;
	case WM_PAINT:
		widget.core.hDC = BeginPaint(hWnd, &paint);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		exposeAbacus(&widget);
		if (widget.abacus.demo) {
			exposeAbacusDemo(&widget);
		}
		(void) EndPaint(hWnd, &paint);
		break;
	case WM_LBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
#if 0
		if (widget.abacus.demo) {
			clearAbacus(&widget);
			clearAbacusDemo(&widget);
		} else
#endif
		{
			selectAbacus(&widget, LOWORD(lParam),
				HIWORD(lParam));
		}
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONUP:
	case WM_NCMOUSEMOVE:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		releaseAbacus(&widget, LOWORD(lParam), HIWORD(lParam));
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
	case WM_MOUSEWHEEL:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		{
			int zDelta = ((short) HIWORD(wParam));
			POINT cursor, origin;

			origin.x = 0, origin.y = 0;
			ClientToScreen(hWnd, &origin);
			(void) GetCursorPos(&cursor);
			if (zDelta > (WHEEL_DELTA >> 1)) {
				moveAbacusInput(&widget, cursor.x - origin.x,
					cursor.y - origin.y, '8',
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
				if (GetKeyState(VK_CONTROL) >> 1) {
					resizeAbacus(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
				}
			} else if (zDelta < -(WHEEL_DELTA >> 1)) {
				moveAbacusInput(&widget, cursor.x - origin.x,
					cursor.y - origin.y, '2',
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
				if (GetKeyState(VK_CONTROL) >> 1) {
					resizeAbacus(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
				}
			}
		}
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#endif
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ACTION_EXIT:
			if (hDemoDlg != NULL) {
				(void) EndDialog(hDemoDlg, TRUE);
				hDemoDlg = NULL;
			}
			if (widget.abacus.demo)
				toggleDemoAbacusDemo(&widget);
			else
				destroyAbacus(&widget, brush);
			break;
		case ACTION_HIDE:
			hideAbacus(&widget);
			break;
		case ACTION_CLEAR:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			clearAbacus(&widget);
			if (widget.abacus.demo) {
				clearAbacusDemo(&widget);
			}
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_RIGHT:
		case ACTION_LEFT:
		case ACTION_UP:
		case ACTION_DOWN:
			{
				int action = LOWORD(wParam);
				POINT cursor, origin;
				char letter;

				if (action == ACTION_RIGHT || action == ACTION_LEFT)
					letter = ((action == ACTION_RIGHT) ? '6' : '4');
				else
					letter = ((action == ACTION_UP) ? '8' : '2');
				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) moveAbacusInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					letter, FALSE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_INCX:
		case ACTION_DECX:
		case ACTION_INCY:
		case ACTION_DECY:
			{
				int action = LOWORD(wParam);
				POINT cursor, origin;
				char letter;

				if (action == ACTION_INCX || action == ACTION_DECX)
					letter = ((action == ACTION_INCX) ? '6' : '4');
				else
					letter = ((action == ACTION_INCY) ? '8' : '2');
				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) moveAbacusInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					letter, TRUE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_UNDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			undoAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_REDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			redoAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_COMPLEMENT:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			complementAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_INCREMENT:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			incrementAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_DECREMENT:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			decrementAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_FORMAT:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			if (ENABLE_SUBDECKS(widget.abacus.decks[BOTTOM].piece, widget.abacus.decks[BOTTOM].piecePercent, widget.abacus.decimalPosition, !widget.abacus.slot, widget.abacus.subbase)) {
				clearDecimalAbacus(&widget);
			}
			changeFormatAbacus(&widget);
			checkEnabled(&widget);
			if (widget.abacus.demo) {
				clearAbacusDemo(&widget);
				clearAbacus(&widget);
			}
			break;
		case ACTION_ROMAN_NUMERALS:
			changeRomanNumeralsAbacus(&widget);
			EnableMenuItem(widget.core.hMenu, ACTION_LATIN,
				(unsigned) (ENABLE_LATIN(widget.abacus.romanNumeralsMode, widget.abacus.decks[BOTTOM].piece) ?
				MF_ENABLED : MF_GRAYED));
			break;
		case ACTION_ROMAN_MARKERS:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			changeRomanMarkersAbacus(&widget);
			checkEnabled(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_LATIN:
			toggleLatinAbacus(&widget);
			break;
		case ACTION_GROUP:
			toggleGroupingAbacus(&widget);
			break;
		case ACTION_PLACE_ON_RAIL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			togglePlaceOnRailAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_VERTICAL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			toggleVerticalAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		/*case ACTION_STIPPLE_FRAME:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			toggleStippleFrameAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;*/
		case ACTION_SIGN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			toggleNegativeSignAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_PIECE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			if (ENABLE_SUBDECKS(((widget.abacus.decks[BOTTOM].piece != 4) ? 4 : 0), widget.abacus.decks[BOTTOM].piecePercent, widget.abacus.decimalPosition, widget.abacus.slot, widget.abacus.subbase)) {
				clearDecimalAbacus(&widget);
			}
			changePieceAbacus(&widget);
			checkEnabled(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_PIECE_PERCENT:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			if (ENABLE_SUBDECKS(widget.abacus.decks[BOTTOM].piece, ((widget.abacus.decks[BOTTOM].piecePercent == 0) ? 4 : 0), widget.abacus.decimalPosition, widget.abacus.slot, widget.abacus.subbase)) {
				clearDecimalAbacus(&widget);
			}
			changePiecePercentAbacus(&widget);
			checkEnabled(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_SUBDECK:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			changeSubdecksAbacus(&widget);
			if (ENABLE_SUBDECKS(widget.abacus.decks[BOTTOM].piece, widget.abacus.decks[BOTTOM].piecePercent, widget.abacus.decimalPosition, widget.abacus.slot, ((widget.abacus.subbase != 0) ? 4 : 0))) {
				clearDecimalAbacus(&widget);
			}
			checkEnabled(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_SUBDECKS_SEPARATED:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			toggleSubdecksSeparatedAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_ALT_SUBBEAD_PLACEMENT:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			toggleAltSubbeadPlacementAbacus(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_MUSEUM:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			changeMuseumAbacus(&widget);
			checkEnabled(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
#ifdef ANOMALY
		case ACTION_ANOMALY:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			changeAnomalyAbacus(&widget);
			checkEnabled();
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
#endif
		case ACTION_RIGHT_TO_LEFT_ADD:
			toggleRightToLeftAddAbacus(&widget);
			break;
		case ACTION_RIGHT_TO_LEFT_MULT:
			toggleRightToLeftMultAbacus(&widget);
			break;
		case ACTION_DEMO:
#define DEMO_DIALOG
#ifdef DEMO_DIALOG
			if (hDemoDlg != NULL) {
				(void) DestroyWindow(hDemoDlg);
				hDemoDlg = NULL;
				widget.abacus.demo = False;
			}
			hDemoDlg = CreateDialog(widget.core.hInstance,
				"Demo", hWnd,
				(DLGPROC) demo);
#endif
			if (!widget.abacus.demo)
				toggleDemoAbacusDemo(&widget);
			resizeAbacus(&widget);
			(void) InvalidateRect(hWnd, NULL,
				TRUE);
			break;
		case ACTION_NEXT:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			/* showNextAbacus(&widget); */
			if (widget.abacus.demo)
				showNextAbacusDemo(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_ENTER:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			/* showEnterAbacus(&widget); */
			if (widget.abacus.demo)
				showEnterAbacusDemo(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_JUMP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			/* showJumpAbacus(&widget); */
			if (widget.abacus.demo)
				showJumpAbacusDemo(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_CHAPTER1:
		case ACTION_CHAPTER2:
		case ACTION_CHAPTER3:
		case ACTION_CHAPTER4:
		case ACTION_CHAPTER5:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			/* showChapterAbacus(&widget); */
			if (widget.abacus.demo)
				showChapterAbacusDemo(&widget,
					(unsigned int) (LOWORD(wParam) - ACTION_CHAPTER1));
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_MORE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			/* showMoreAbacus(&widget); */
			if (widget.abacus.demo)
				showMoreAbacusDemo(&widget);
			(void) ReleaseDC(hWnd,
				widget.core.hDC);
			break;
		case ACTION_SOUND:
			toggleSoundAbacus(&widget);
			break;
		case ACTION_SPEED_UP:
			speedUpAbacus(&widget);
			break;
		case ACTION_SLOW_DOWN:
			slowDownAbacus(&widget);
			break;
		case ACTION_CALC:
			if (hCalcDlg != NULL) {
				(void) DestroyWindow(hCalcDlg);
				hCalcDlg = NULL;
			}
			hCalcDlg = CreateDialog(widget.core.hInstance,
				"Calculate", hWnd,
				(DLGPROC) calculation);
			break;
		case ACTION_TEACH:
			/*(void) DialogBox(widget.core.hInstance,
				"Teach", hWnd,
				(DLGPROC) teach);*/
			forceTeachParams(&widget);
			if (hTeachDlg != NULL) {
				(void) DestroyWindow(hTeachDlg);
				hTeachDlg = NULL;
				widget.abacus.teach = False;
			}
			hTeachDlg = CreateDialog(widget.core.hInstance,
				"Teach", hWnd,
				(DLGPROC) teach);
			break;
		case ACTION_DESCRIPTION:
			strncpy(descriptionHelp, description1Help, DESCRIPTION1_SIZE);
			strncat(descriptionHelp, "\n", 2);
			strncat(descriptionHelp, description2Help, DESCRIPTION2_SIZE);
			(void) MessageBox(hWnd, descriptionHelp,
				"Description", MB_OK | MB_ICONQUESTION);
			break;
		case ACTION_FEATURES:
			(void) MessageBox(hWnd, featuresHelp,
				"Features", MB_OK | MB_ICONEXCLAMATION);
			break;
		case ACTION_REFERENCES:
			(void) MessageBox(hWnd, referencesHelp,
				"References", MB_OK | MB_ICONINFORMATION);
			break;
		case ACTION_ABOUT:
			(void) DialogBox(widget.core.hInstance,
				"About", hWnd, (DLGPROC) about);
			break;
		}
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

void drawDemoText(const char* line, int i)
{
	if (hDemoDlg == NULL || line == NULL)
		return;
	SetDlgItemText(hDemoDlg, ACTION_DEMO1 + i, line);
}

void drawTeachText(const char* line, int i)
{
	if (hTeachDlg == NULL || line == NULL)
		return;
	SetDlgItemText(hTeachDlg, ACTION_TEACH1 + i, line);
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(XtPointer) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0)) {
		if (!IsDialogMessage(hCalcDlg, &msg) &&
				!IsDialogMessage(hTeachDlg, &msg)) {
			if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
				(void) TranslateMessage(&msg);
				(void) DispatchMessage(&msg);
			}
		}
	}
	return (int) msg.wParam;
}

#else
#include "xwin.h"
#include "xgui.h"
#include "Abacus.h"
#ifdef HAVE_XPM
#ifdef CONSTPIXMAPS
#include "icons/16x16/abacus.xpm"
#include "icons/22x22/abacus.xpm"
#include "icons/24x24/abacus.xpm"
#include "icons/32x32/abacus.xpm"
#include "icons/48x48/abacus.xpm"
#include "icons/64x64/abacus.xpm"
#else
#include "pixmaps/16x16/abacus.xpm"
#include "pixmaps/22x22/abacus.xpm"
#include "pixmaps/24x24/abacus.xpm"
#include "pixmaps/32x32/abacus.xpm"
#include "pixmaps/48x48/abacus.xpm"
#include "pixmaps/64x64/abacus.xpm"
#endif
#define RESIZE_XPM(s) ((char **) (((s)<=32)?\
(((s)<=22)?(((s)<=16)?abacus_16x16:abacus_22x22):\
(((s)<=24)?abacus_24x24:abacus_32x32)):\
(((s)<=48)?abacus_48x48:abacus_64x64)))
#endif
#include "pixmaps/64x64/abacus.xbm"
#define DEFINE_XBM (char *) abacus_64x64_bits, abacus_64x64_width, abacus_64x64_height
#ifdef HAVE_EDITRES
#ifdef __VMS
#include <Xmu/Editres.h>
#else
#include <X11/Xmu/Editres.h>
#endif
#endif

#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
#ifdef LEE_ABACUS
#undef LEE_ABACUS
#endif
#define MAX_DIGITS 256		/* This limits the number of rails */
/*#define TITLE_LENGTH (MAX_DIGITS+FILE_NAME_LENGTH+3)*/
#endif

#ifdef HAVE_MOTIF
static Widget formatMenu;
static Widget romanNumeralsMenu, romanMarkersMenu;
static Widget pieceRailMenu, piecePercentRailMenu;
static Widget subdeckRailMenu, museumMenu;
#ifdef ANOMALY
static Widget anomalyMenu;
#endif
static Widget teachRowCol;
static Widget formatSubMenu;
static Widget formatOptions[MAX_MODES];
static Widget secondaryRailsMenu = None;
static Widget demoToggle = None, teachToggle = None;
static Widget baseDialog = None, delayDialog = None;
static Widget pauseDialog;
static char *mathBuffer = NULL;
#elif defined(HAVE_ATHENA)
static Widget railSliderLabel, formatLabel;
static Widget romanNumeralsLabel, romanMarkersLabel;
static Widget pieceRailLabel, piecePercentRailLabel;
static Widget subdeckRailLabel, museumLabel;
static Widget demoMenuItem;
#ifdef ANOMALY
static Widget anomalyLabel;
#endif
static Widget delaySliderLabel;
static Widget abacusBaseSliderLabel, displayBaseSliderLabel;
static Widget latinBox;
static Widget displayDialogShell, specialRailsDialogShell;
static Widget secondaryRailsDialogShell, setupDialogShell;
static Widget calculateDialogShell, teachDialogShell;
static Widget learnMenu;
static const char *fileTypes[] =
{
	"Exit"
};
#define numFileTypes (sizeof(fileTypes)/sizeof(fileTypes[0]))
static const char *controlTypes[] =
{
	"Clear",
	"Complement -",
	"Undo",
	"Redo",
	"Display...",
	"Special Rails...",
	"Secondary Rails...",
	"Setup..."
};
#define numControlTypes (sizeof(controlTypes)/sizeof(controlTypes[0]))
static const char *learnTypes[] =
{
	"Calculate...",
	"Demo",
	"Teach..."
};
#define numLearnTypes (sizeof(learnTypes)/sizeof(learnTypes[0]))
static const char *helpTypes[] =
{
	"Description...",
	"Features...",
	"Synopsis...",
	"Options...",
	"References...",
	"About..."
};
#define numHelpTypes (sizeof(helpTypes)/sizeof(helpTypes[0]))
#endif
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
extern void initializeText(AbacusWidget w);
#ifdef USE_SPIN
	Boolean spin = True;
#else
	Boolean spin = False;
#endif
#ifdef USE_BASE_SPIN
	Boolean baseSpin = True;
#else
	Boolean baseSpin = False;
#endif
#define MAX_RAILS 24		/* Arbitrary but lower for precision of slider */
#define MAX_DELAY 500
static Widget mainPanel, railSlider;
static Widget latinToggle, groupToggle, placeOnRailToggle, verticalToggle;
static Widget stippleFrameToggle, signToggle;
static Widget subdecksSeparatedToggle, altSubbeadPlacementToggle;
static Widget displayBaseSlider, abacusBaseSlider;
static Widget delaySlider;
static Widget soundToggle;
static Widget rightToLeftAddToggle, rightToLeftMultToggle;
static Widget clearDialog;
static Widget descriptionDialog, featuresDialog;
static Widget synopsisDialog;
static Widget optionsDialog;
static Widget referencesDialog, aboutDialog;
static Widget tracker;
#ifdef LEE_ABACUS
static Widget leftAuxAbacus = NULL, rightAuxAbacus = NULL;
static Widget leftAuxTracker = NULL, rightAuxTracker = NULL;
#endif
static Widget teachTracker = NULL, teachLine[3];
static char *railLabel = (char *) "Abacus Size:";
static char *displayBaseLabel = (char *) "Display Base:";
static char *abacusBaseLabel = (char *) "Abacus Base: ";
static char *delayLabel = (char *) "msec Delay:";
#else
static Widget shell = NULL;
static char titleDsp[TITLE_LENGTH];
#endif
static Pixmap pixmap = None;
static Widget topLevel, abacus, abacusDemo = NULL;

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*abacus.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*abacus.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*abacus.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*abacus.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*abacus.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*abacus.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-transparent", (char *) "*abacus.transparent", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-notransparent", (char *) "*abacus.transparent", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-foreground", (char *) "*abacus.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-fg", (char *) "*abacus.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*abacus.borderColor", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*abacus.borderColor", XrmoptionSepArg, NULL},
	{(char *) "-frame", (char *) "*abacus.frameColor", XrmoptionSepArg, NULL},
	{(char *) "-primaryBeadColor", (char *) "*abacus.primaryBeadColor", XrmoptionSepArg, NULL},
	{(char *) "-leftAuxBeadColor", (char *) "*abacus.leftAuxBeadColor", XrmoptionSepArg, NULL},
	{(char *) "-rightAuxBeadColor", (char *) "*abacus.rightAuxBeadColor", XrmoptionSepArg, NULL},
	{(char *) "-secondaryBeadColor", (char *) "*abacus.secondaryBeadColor", XrmoptionSepArg, NULL},
	{(char *) "-highlightBeadColor", (char *) "*abacus.highlightBeadColor", XrmoptionSepArg, NULL},
	{(char *) "-primaryMarkerColor", (char *) "*abacus.primaryMarkerColor", XrmoptionSepArg, NULL},
	{(char *) "-leftAuxMarkerColor", (char *) "*abacus.leftAuxMarkerColor", XrmoptionSepArg, NULL},
	{(char *) "-rightAuxMarkerColor", (char *) "*abacus.rightAuxMarkerColor", XrmoptionSepArg, NULL},
	{(char *) "-primaryRailColor", (char *) "*abacus.primaryRailColor", XrmoptionSepArg, NULL},
	{(char *) "-secondaryRailColor", (char *) "*abacus.secondaryRailColor", XrmoptionSepArg, NULL},
	{(char *) "-highlightRailColor", (char *) "*abacus.highlightRailColor", XrmoptionSepArg, NULL},
	{(char *) "-lineRailColor", (char *) "*abacus.lineRailColor", XrmoptionSepArg, NULL},
	{(char *) "-bumpSound", (char *) "*abacus.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-moveSound", (char *) "*abacus.moveSound", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*abacus.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*abacus.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-delay", (char *) "*abacus.delay", XrmoptionSepArg, NULL},
	{(char *) "-demo", (char *) "*abacus.demo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nodemo", (char *) "*abacus.demo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-script", (char *) "*abacus.script", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noscript", (char *) "*abacus.script", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-demopath", (char *) "*abacus.demoPath", XrmoptionSepArg, NULL},
	{(char *) "-demofont", (char *) "*abacus.demoFont", XrmoptionSepArg, NULL},
	{(char *) "-demofn", (char *) "*abacus.demoFont", XrmoptionSepArg, NULL},
	{(char *) "-demoforeground", (char *) "*abacus.demoForeground", XrmoptionSepArg, NULL},
	{(char *) "-demofg", (char *) "*abacus.demoForeground", XrmoptionSepArg, NULL},
	{(char *) "-demobackground", (char *) "*abacus.demoBackground", XrmoptionSepArg, NULL},
	{(char *) "-demobg", (char *) "*abacus.demoBackground", XrmoptionSepArg, NULL},
	{(char *) "-teach", (char *) "*abacus.teach", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noteach", (char *) "*abacus.teach", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rightToLeftAdd", (char *) "*abacus.rightToLeftAdd", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norightToLeftAdd", (char *) "*abacus.rightToLeftAdd", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rightToLeftMult", (char *) "*abacus.rightToLeftMult", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norightToLeftMult", (char *) "*abacus.rightToLeftMult", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-lee", (char *) "*abacus.lee", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nolee", (char *) "*abacus.lee", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rails", (char *) "*abacus.rails", XrmoptionSepArg, NULL},
	{(char *) "-rightAuxRails", (char *) "*abacus.rightAuxRails", XrmoptionSepArg, NULL},
	{(char *) "-leftAuxRails", (char *) "*abacus.leftAuxRails", XrmoptionSepArg, NULL},
	{(char *) "-vertical", (char *) "*abacus.vertical", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-novertical", (char *) "*abacus.vertical", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-stippleFrame", (char *) "*abacus.stippleFrame", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nostippleFrame", (char *) "*abacus.stippleFrame", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-colorScheme", (char *) "*abacus.colorScheme", XrmoptionSepArg, NULL},
	{(char *) "-slot", (char *) "*abacus.slot", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noslot", (char *) "*abacus.slot", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-diamond", (char *) "*abacus.diamond", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nodiamond", (char *) "*abacus.diamond", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-railIndex", (char *) "*abacus.railIndex", XrmoptionSepArg, NULL},
	{(char *) "-topOrient", (char *) "*abacus.topOrient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-notopOrient", (char *) "*abacus.topOrient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bottomOrient", (char *) "*abacus.bottomOrient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nobottomOrient", (char *) "*abacus.bottomOrient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-topNumber", (char *) "*abacus.topNumber", XrmoptionSepArg, NULL},
	{(char *) "-bottomNumber", (char *) "*abacus.bottomNumber", XrmoptionSepArg, NULL},
	{(char *) "-topFactor", (char *) "*abacus.topFactor", XrmoptionSepArg, NULL},
	{(char *) "-bottomFactor", (char *) "*abacus.bottomFactor", XrmoptionSepArg, NULL},
	{(char *) "-topSpaces", (char *) "*abacus.topSpaces", XrmoptionSepArg, NULL},
	{(char *) "-bottomSpaces", (char *) "*abacus.bottomSpaces", XrmoptionSepArg, NULL},
	{(char *) "-topPiece", (char *) "*abacus.topPiece", XrmoptionSepArg, NULL},
	{(char *) "-bottomPiece", (char *) "*abacus.bottomPiece", XrmoptionSepArg, NULL},
	{(char *) "-topPiecePercent", (char *) "*abacus.topPiecePercent", XrmoptionSepArg, NULL},
	{(char *) "-bottomPiecePercent", (char *) "*abacus.bottomPiecePercent", XrmoptionSepArg, NULL},
	{(char *) "-shiftPercent", (char *) "*abacus.shiftPercent", XrmoptionSepArg, NULL},
	{(char *) "-subdeck", (char *) "*abacus.subdeck", XrmoptionSepArg, NULL},
	{(char *) "-subbead", (char *) "*abacus.subbead", XrmoptionSepArg, NULL},
	{(char *) "-sign", (char *) "*abacus.sign", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosign", (char *) "*abacus.sign", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-decimalPosition", (char *) "*abacus.decimalPosition", XrmoptionSepArg, NULL},
	{(char *) "-group", (char *) "*abacus.group", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nogroup", (char *) "*abacus.group", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-groupSize", (char *) "*abacus.groupSize", XrmoptionSepArg, NULL},
	{(char *) "-placeOnRail", (char *) "*abacus.placeOnRail", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noplaceOnRail", (char *) "*abacus.placeOnRail", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-decimalComma", (char *) "*abacus.decimalComma", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nodecimalComma", (char *) "*abacus.decimalComma", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-base", (char *) "*abacus.base", XrmoptionSepArg, NULL},
	{(char *) "-subbase", (char *) "*abacus.subbase", XrmoptionSepArg, (char *) NULL},
	{(char *) "-anomaly", (char *) "*abacus.anomaly", XrmoptionSepArg, NULL},
	{(char *) "-shiftAnomaly", (char *) "*abacus.shiftAnomaly", XrmoptionSepArg, NULL},
	{(char *) "-anomalySq", (char *) "*abacus.anomalySq", XrmoptionSepArg, NULL},
	{(char *) "-shiftAnomalySq", (char *) "*abacus.shiftAnomalySq", XrmoptionSepArg, NULL},
	{(char *) "-displayBase", (char *) "*abacus.displayBase", XrmoptionSepArg, NULL},
	{(char *) "-pressOffset", (char *) "*abacus.pressOffset", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopressOffset", (char *) "*abacus.pressOffset", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-romanNumerals", (char *) "*abacus.romanNumerals", XrmoptionSepArg, (char *) NULL},
	{(char *) "-latin", (char *) "*abacus.latin", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nolatin", (char *) "*abacus.latin", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-romanMarkers", (char *) "*abacus.romanMarkers", XrmoptionSepArg, (char *) NULL},
	{(char *) "-subdecksSeparated", (char *) "*abacus.subdecksSeparated", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosubdecksSeparated", (char *) "*abacus.subdecksSeparated", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-altSubbeadPlacement", (char *) "*abacus.altSubbeadPlacement", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noaltSubbeadPlacement", (char *) "*abacus.altSubbeadPlacement", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-chinese", (char *) "*abacus.format", XrmoptionNoArg, (char *) "chinese"},
	{(char *) "-japanese", (char *) "*abacus.format", XrmoptionNoArg, (char *) "japanese"},
	{(char *) "-korean", (char *) "*abacus.format", XrmoptionNoArg, (char *) "korean"},
	{(char *) "-roman", (char *) "*abacus.format", XrmoptionNoArg, (char *) "roman"},
	{(char *) "-russian", (char *) "*abacus.format", XrmoptionNoArg, (char *) "russian"},
	{(char *) "-danish", (char *) "*abacus.format", XrmoptionNoArg, (char *) "danish"},
	{(char *) "-medieval", (char *) "*abacus.format", XrmoptionNoArg, (char *) "medieval"},
	{(char *) "-generic", (char *) "*abacus.format", XrmoptionNoArg, (char *) "generic"},
	{(char *) "-museum", (char *) "*abacus.museum", XrmoptionSepArg, (char *) "--"},
	{(char *) "-version", (char *) "*abacus.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef LEE_ABACUS
#define AUXWIN(n) (n == LEFT_AUX) ? ((leftAuxAbacus) ? leftAuxAbacus : abacus) : ((n == RIGHT_AUX ) ? ((rightAuxAbacus) ? rightAuxAbacus : abacus) : abacus)
#else
#define AUXWIN(n) abacus
#endif

static void printState(Widget w, char *msg
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
	, const char *concatString
#endif
	)
{
#ifdef HAVE_MOTIF
	XtVaSetValues(w,
		XmNvalue, msg, NULL);
/* label widget and draw the strings on it.
  use the (void) XmStringDrawImage(XtDisplay(topLevel),
			XtWindow(topLevel), fontlist, (char *) "Primary",
			gc, x, y, width,
			XmALIGNMENT_BEGINNING, XmSTRING_DIRECTION_L_TO_R, NULL); */
#elif defined(HAVE_ATHENA)
	XtVaSetValues(w,
		XtNlabel, msg, NULL);
#else
	concatTitle(titleDsp, msg, " ", concatString, TITLE_LENGTH);
	XtVaSetValues(w,
		XtNtitle, titleDsp, NULL);
#endif
}

/* There's probably a better way to assure that they are the same but I do
 * not know it off hand. */
static void
initializeDemo(void)
{
	int mode;
	Boolean mono, reverse;
	Pixel demoForeground, demoBackground;
	String demoPath, demoFont;

	XtVaGetValues(abacus,
		XtNmono, &mono,
		XtNreverseVideo, &reverse,
		XtNmode, &mode,
		XtNdemoForeground, &demoForeground,
		XtNdemoBackground, &demoBackground,
		XtNdemoPath, &demoPath,
		XtNdemoFont, &demoFont, NULL);
	XtVaSetValues(abacusDemo,
		XtNmono, mono,
		XtNreverseVideo, reverse,
		XtNmode, mode,
		XtNdemoForeground, demoForeground,
		XtNdemoBackground, demoBackground,
		XtNdemoPath, demoPath,
		XtNdemoFont, demoFont,
		XtNframed, True, NULL);
}

#ifdef LEE_ABACUS
static void
initializeAux(void)
{
	Boolean mono, reverse, script, stippleFrame;
	Pixel frameColor;

	XtVaGetValues(abacus,
		XtNmono, &mono,
		XtNreverseVideo, &reverse,
		XtNscript, &script,
		XtNframeColor, &frameColor,
		XtNstippleFrame, &stippleFrame, NULL);
	if (leftAuxAbacus) {
		XtVaSetValues(leftAuxAbacus,
			XtNmono, mono,
			XtNreverseVideo, reverse,
			XtNscript, script,
			XtNframeColor, frameColor,
			XtNstippleFrame, stippleFrame, NULL);
		XtVaSetValues(abacus,
			XtNleftAuxAbacus, leftAuxAbacus, NULL);
	}
	if (rightAuxAbacus) {
		XtVaSetValues(rightAuxAbacus,
			XtNmono, mono,
			XtNreverseVideo, reverse,
			XtNscript, script,
			XtNframeColor, frameColor,
			XtNstippleFrame, stippleFrame, NULL);
		XtVaSetValues(abacus,
			XtNrightAuxAbacus, rightAuxAbacus, NULL);
	}
}
#endif

#ifdef HAVE_ATHENA
static void
setDemoCheck(Widget w)
{
	Boolean demo;
	XtVaGetValues(abacus,
		XtNdemo, &demo, NULL);
	menuCheck(topLevel, w, demo);
}
#endif

static void
abacusDemoCallback(Widget w, caddr_t clientData, abacusCallbackStruct *callData)
{
#ifndef HAVE_MOTIF
	int mode;

	XtVaGetValues(w,
		XtNmode, &mode, NULL);
#endif
	switch (callData->reason) {
	case ACTION_BASE_DEFAULT:
#ifdef HAVE_MOTIF
		setChanger(abacusBaseSlider, DEFAULT_BASE, baseSpin);
#endif
		break;
	case ACTION_HIDE:
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		(void) XIconifyWindow(XtDisplay(shell),
			XtWindow(shell),
			XScreenNumberOfScreen(XtScreen(shell)));
#endif
		break;
	case ACTION_MOVE:
#ifndef LEE_ABACUS
		if (callData->aux == PRIMARY)
#endif
		{
			XtVaSetValues(AUXWIN(callData->aux),
				XtNdeck, callData->deck,
				XtNrail, callData->rail,
				XtNnumber, callData->number, NULL);
		}
		break;
	case ACTION_HIGHLIGHT_RAIL:
#ifndef LEE_ABACUS
		if (callData->aux == PRIMARY)
#endif
		{
			XtVaSetValues(AUXWIN(callData->aux),
				XtNdeck, HIGHLIGHT_DECK,
				XtNrail, callData->rail, NULL);
		}
		break;
	case ACTION_UNHIGHLIGHT_RAIL:
#ifndef LEE_ABACUS
		if (callData->aux == PRIMARY)
#endif
		{
			XtVaSetValues(AUXWIN(callData->aux),
				XtNdeck, UNHIGHLIGHT_DECK,
				XtNrail, callData->rail, NULL);
		}
		break;
	case ACTION_HIGHLIGHT_RAILS:
#ifndef LEE_ABACUS
		if (callData->aux == PRIMARY)
#endif
		{
			XtVaSetValues(AUXWIN(callData->aux),
				XtNdeck, HIGHLIGHTS_DECK, NULL);
		}
		break;
	case ACTION_CLEAR_NODEMO:
	case ACTION_CLEAR:
		XtVaSetValues(abacus,
			XtNdeck, CLEAR_DECK, NULL);
#ifdef LEE_ABACUS
		if (leftAuxAbacus)
			XtVaSetValues(leftAuxAbacus,
				XtNdeck, CLEAR_DECK, NULL);
		if (rightAuxAbacus)
			XtVaSetValues(rightAuxAbacus,
				XtNdeck, CLEAR_DECK, NULL);
#endif
		break;
	case ACTION_DECIMAL_CLEAR:
		XtVaSetValues(abacus,
			XtNdeck, CLEAR_DECIMAL_DECK, NULL);
		break;
	case ACTION_DEMO:
		XtDestroyWidget(abacusDemo);
		XtVaSetValues(abacus,
			XtNdemo, False, NULL);
#ifdef HAVE_MOTIF
		XtVaSetValues(demoToggle,
			XmNset, False, NULL);
		/*XtVaSetValues(choiceMenu,
			XmNmenuHistory, choiceOptions[NORMAL], NULL);*/
#elif defined(HAVE_ATHENA)
		setDemoCheck(demoMenuItem);
#else
		/* Destroying shell is causing trouble, so lets not do that */
		/* Warning: XtRemoveGrab asked to remove a widget not on the list */
		/* http://www.mail-archive.com/lesstif@hungry.com/msg00535.html */
		/*XtDestroyWidget(shell);
		shell = NULL;*/
		XtUnrealizeWidget(shell);
#endif
#ifdef LEE_ABACUS
		if (leftAuxAbacus)
			XtVaSetValues(leftAuxAbacus,
				XtNdemo, False, NULL);
		if (rightAuxAbacus)
			XtVaSetValues(rightAuxAbacus,
				XtNdemo, False, NULL);
#endif
		break;
	}
	if (callData->reason == ACTION_SCRIPT || callData->reason == ACTION_IGNORE) {
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
#ifdef LEE_ABACUS
		if (leftAuxTracker && leftAuxAbacus && w == leftAuxAbacus)
			printState(leftAuxTracker, callData->buffer);
		else if (rightAuxTracker && rightAuxAbacus && w == rightAuxAbacus)
			printState(rightAuxTracker, callData->buffer);
		else
#endif
			printState(tracker, callData->buffer);
#else
		printState(XtParent(w), callData->buffer,
			((mode < 0 || mode >= MAX_FORMATS)
			? "Abacus" : formatChoices[mode]));
#endif
	}
}

static void
forceNormalParams(Widget w)
{
	int anomaly, anomalySq, base, displayBase;

	XtVaGetValues(w,
		XtNanomaly, &anomaly,
		XtNanomalySq, &anomalySq,
		XtNbase, &base,
		XtNdisplayBase, &displayBase, NULL);
	if (anomaly != 0)
		XtVaSetValues(w,
			XtNanomaly, 0, NULL);
	if (anomalySq != 0)
		XtVaSetValues(w,
			XtNanomalySq, 0, NULL);
	if (base != DEFAULT_BASE) {
		base = DEFAULT_BASE;
		XtVaSetValues(w,
			XtNbase, base, NULL);
#ifdef HAVE_MOTIF
		setChanger(abacusBaseSlider, base, baseSpin);
#endif
	}
	if (displayBase != DEFAULT_BASE) {
		displayBase = DEFAULT_BASE;
		XtVaSetValues(w,
			XtNdisplayBase, displayBase, NULL);
#ifdef HAVE_MOTIF
		setChanger(displayBaseSlider, displayBase, baseSpin);
#endif
	}
#ifdef LEE_ABACUS
	if (leftAuxAbacus)
		XtVaSetValues(leftAuxAbacus,
			XtNbase, base,
			XtNdisplayBase, displayBase, NULL);
	if (rightAuxAbacus)
		XtVaSetValues(rightAuxAbacus,
			XtNbase, base,
			XtNdisplayBase, displayBase, NULL);
#endif
}

static void
forceDemoParams(Widget w)
{
	int rails, min, mode;
	int bottomPiece, bottomPiecePercent, shiftPercent;
	Boolean demo, teach, sign;

	forceNormalParams(w);
	XtVaGetValues(abacus,
		XtNmode, &mode,
		XtNrails, &rails,
		XtNsign, &sign,
		XtNbottomPiece, &bottomPiece,
		XtNbottomPiecePercent, &bottomPiecePercent,
		XtNshiftPercent, &shiftPercent,
		XtNdemo, &demo,
		XtNteach, &teach, NULL);
	if (mode == GENERIC) {
		mode = CHINESE;
		XtVaSetValues(abacus,
			XtNmode, mode, NULL);
#ifdef HAVE_MOTIF
		XtVaSetValues(formatMenu,
			XmNmenuHistory, formatOptions[mode], NULL);
#endif
	}
	min = ((sign) ? 1: 0) + ((bottomPiece != 0) ? 1 : 0) +
		((bottomPiecePercent != 0) ? 1 + shiftPercent : 0) +
		((demo || teach) ? MIN_DEMO_RAILS : MIN_RAILS);
	if (rails < min) {
#ifdef HAVE_MOTIF
		setChanger(railSlider, min, spin);
#endif
		XtVaSetValues(abacus,
			XtNrails, min, NULL);
	}
}

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static void
forceTeachParams(Widget w)
{
	int anomaly, anomalySq, base, displayBase;
	Boolean sign;

	XtVaGetValues(w,
		XtNsign, &sign,
		XtNanomaly, &anomaly,
		XtNanomalySq, &anomalySq,
		XtNbase, &base,
		XtNdisplayBase, &displayBase, NULL);
	if (sign)
		XtVaSetValues(w,
			XtNsign, False, NULL);
	if (anomaly != 0)
		XtVaSetValues(w,
			XtNanomaly, 0, NULL);
	if (anomalySq != 0)
		XtVaSetValues(w,
			XtNanomalySq, 0, NULL);
/* allowing them not equal is experimental */
#ifndef BASE_TEACH
	if (displayBase != base)
		XtVaSetValues(w,
			XtNdisplayBase, base, NULL);
#endif
#ifdef LEE_ABACUS
	if (leftAuxAbacus)
		XtVaSetValues(leftAuxAbacus,
			XtNbase, base,
			XtNdisplayBase, displayBase, NULL);
	if (rightAuxAbacus)
		XtVaSetValues(rightAuxAbacus,
			XtNbase, base,
			XtNdisplayBase, displayBase, NULL);
#endif
	/*if (mode == GENERIC) {
		mode = CHINESE;
		XtVaSetValues(w,
			XtNmode, mode, NULL);
		XtVaSetValues(formatMenu,
			XmNmenuHistory, formatOptions[mode], NULL);
	}*/
}
#endif

static void
initialize(Widget w)
{
	Boolean versionOnly;
#ifdef HAVE_ATHENA
	Boolean sound;
#endif

	XtVaGetValues(w,
#ifdef HAVE_ATHENA
		XtNsound, &sound,
#endif
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		printVersionOnly(aboutHelp);
	}
#ifdef HAVE_ATHENA
	XtVaSetValues(soundToggle,
		XtNstate, sound, NULL);
#endif
#ifdef LEE_ABACUS
	initializeAux();
#endif
}

static void
createDemo(void)
{
	char titleDsp[TITLE_FILE_LENGTH];
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	/* This is to work around bug in LessTif # 2801540 */
	/* Cannot really use LESSTIF_VERSION as one may swap out lib */
	int height, subheight = 134, newHeight = 0;

	if (XtIsRealized(topLevel))
		XtVaGetValues(topLevel,
			XtNheight, &height, NULL);
	concatTitle(titleDsp, progDsp, ": ", "Demo", TITLE_FILE_LENGTH);
	abacusDemo = XtVaCreateManagedWidget(titleDsp,
		abacusDemoWidgetClass, mainPanel,
#ifdef HAVE_ATHENA
		XtNfromVert, abacus,
#endif
		NULL);
	if (XtIsRealized(topLevel)) {
		XtVaGetValues(abacusDemo,
			XtNheight, &newHeight, NULL);
		if (newHeight <= 1) {
			XtVaSetValues(topLevel,
				XtNheight, height + subheight, NULL);
		}
#ifdef HAVE_ATHENA
		/* Not quite right but better */
		XtUnrealizeWidget(topLevel);
		XtRealizeWidget(topLevel);
#endif
	}
#else
	concatTitle(titleDsp, progDsp, ": ", "Demo", TITLE_FILE_LENGTH);
	if (shell == NULL)
		shell = XtAppCreateShell(progDsp, titleDsp,
			topLevelShellWidgetClass, XtDisplay(topLevel), NULL, 0);
	XtVaSetValues(shell,
		XtNiconPixmap, pixmap,
		XtNinput, True,
		XtNtitle, titleDsp, NULL);
	abacusDemo = XtVaCreateManagedWidget("abacusDemo",
		abacusDemoWidgetClass, shell, NULL);
#endif
	XtAddCallback(abacusDemo, XtNselectCallback,
		(XtCallbackProc) abacusDemoCallback,
		(XtPointer) NULL);
	initializeDemo();
}

static void
realizeDemo(void)
{
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
	XtRealizeWidget(shell);
#endif
#ifdef HAVE_ATHENA
	{
		int height, subheight = 134, newHeight = 0;

		XtVaGetValues(abacusDemo,
			XtNheight, &newHeight, NULL);
		if (newHeight <= 8) {
			XtVaGetValues(topLevel,
				XtNheight, &height, NULL);
			XtVaSetValues(topLevel,
				XtNheight, height + subheight, NULL);
		}
	}
#endif
	XGrabButton(XtDisplay(abacusDemo), (unsigned int) AnyButton,
		AnyModifier, XtWindow(abacusDemo), TRUE,
		(unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(abacusDemo),
		XCreateFontCursor(XtDisplay(abacusDemo), XC_hand2));
}

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static void
checkEnabled(Widget w) {
	int decimalPosition;
	Boolean slot;
	int bottomPiece, bottomPiecePercent, subbase;
	int romanNumeralsMode, romanMarkersMode;
	int anomaly, anomalySq;

	XtVaGetValues(w,
		XtNslot, &slot,
		XtNdecimalPosition, &decimalPosition,
		XtNbottomPiece, &bottomPiece,
		XtNbottomPiecePercent, &bottomPiecePercent,
		XtNsubbase, &subbase,
		XtNromanNumeralsMode, &romanNumeralsMode,
		XtNromanMarkersMode, &romanMarkersMode,
		XtNanomaly, &anomaly,
		XtNanomalySq, &anomalySq, NULL);

#if 0
	(void) printf("checkEnabled %d %d %d %d %d %d %d %d\n",
			bottomPiece, bottomPiecePercent, decimalPosition,
			slot, subbase, romanMarkersMode, anomaly, anomalySq);
#endif
#ifdef HAVE_MOTIF
	XtSetSensitive(latinToggle,
		ENABLE_LATIN(romanNumeralsMode, bottomPiece));
	XtSetSensitive(piecePercentRailMenu,
		ENABLE_PIECE_PERCENT(bottomPiece, bottomPiecePercent, decimalPosition));
	XtSetSensitive(secondaryRailsMenu,
		ENABLE_SECONDARY(bottomPiece, bottomPiecePercent, decimalPosition, slot));
	XtSetSensitive(subdecksSeparatedToggle,
		ENABLE_SUBDECKS(bottomPiece, bottomPiecePercent, decimalPosition, slot, subbase));
	XtSetSensitive(altSubbeadPlacementToggle,
		ENABLE_SUBDECKS(bottomPiece, bottomPiecePercent, decimalPosition, slot, subbase));
	XtSetSensitive(museumMenu,
		ENABLE_MUSEUM(bottomPiece, bottomPiecePercent, decimalPosition, slot, subbase, romanMarkersMode, anomaly, anomalySq));
	XtSetSensitive(romanMarkersMenu,
		ENABLE_ROMAN_MARKERS(anomaly, anomalySq));
#elif defined(HAVE_ATHENA)
	XtSetSensitive(latinToggle,
		ENABLE_LATIN(romanNumeralsMode, bottomPiece));
#endif
}
#endif

#ifdef HAVE_MOTIF
static void
abacusTeachCallback(Widget w, caddr_t clientData,
		abacusCallbackStruct *callData)
{
	Boolean demo;
	char * str = XmTextGetString(w);

	XtVaGetValues(abacus,
		XtNdemo, &demo, NULL);
	if (demo)
		return;
	if (str != NULL && strlen((char *) str) > 0) {
		XtVaSetValues(abacus,
			XtNdeck, TEACH_DECK,
			XtNteachBuffer, str, NULL);
		free(str);
	}
}

static void
createTeach(Widget w)
{
	int columns = CALC_STRING_SIZE, subheight = 100;
	/* This is to work around bug in LessTif # 2801540 */
	/* Cannot really use LESSTIF_VERSION as one may swap out lib */
	int height = 1, newHeight = 1;

	if (XtIsRealized(topLevel))
		XtVaGetValues(topLevel,
			XtNheight, &height, NULL);
	teachRowCol = XtVaCreateManagedWidget("teachRowCol",
		xmRowColumnWidgetClass, mainPanel,
		XmNborderWidth, 1,
		XmNheight, subheight, NULL);
	if (XtIsRealized(topLevel)) {
		XtVaGetValues(teachRowCol,
			XtNheight, &newHeight, NULL);
		if (newHeight <= 1)
			XtVaSetValues(topLevel, XtNheight, height + subheight, NULL);
	}

	teachTracker = XtVaCreateManagedWidget("0",
		xmTextWidgetClass, teachRowCol, NULL);
	teachLine[0] = XtVaCreateManagedWidget("teachText1",
		xmLabelGadgetClass, teachRowCol,
		XtVaTypedArg, XmNlabelString,
		XmRString, " ", columns, NULL);
	teachLine[1] = XtVaCreateManagedWidget("teachText2",
		xmLabelGadgetClass, teachRowCol,
		XtVaTypedArg, XmNlabelString,
		XmRString, " ", columns, NULL);
	teachLine[2] = XtVaCreateManagedWidget("teachText3",
		xmLabelGadgetClass, teachRowCol,
		XtVaTypedArg, XmNlabelString,
		XmRString, " ", columns, NULL);
	XtAddCallback(teachTracker, XmNactivateCallback,
		(XtCallbackProc) abacusTeachCallback,
		(XtPointer) NULL);
	initializeText((AbacusWidget) w);
}

static void
destroyTeach(void)
{
	XtDestroyWidget(teachTracker);
	XtDestroyWidget(teachLine[0]);
	XtDestroyWidget(teachLine[1]);
	XtDestroyWidget(teachLine[2]);
	XtDestroyWidget(teachRowCol);
	teachTracker = NULL;
}
#endif

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
#ifdef LEE_ABACUS
extern Pixel darker(AbacusWidget w, Pixel pixel);
#endif

static void
latinCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#else
		Boolean state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif
	XtVaSetValues(abacus,
		XtNlatin, state, NULL);
}

static void
groupCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#else
		Boolean state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif
	XtVaSetValues(abacus,
		XtNgroup, state, NULL);
}

static void
placeOnRailCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#else
		Boolean state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif
	XtVaSetValues(abacus,
		XtNplaceOnRail, state, NULL);
}

static void
verticalCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#else
		Boolean state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif
	XtVaSetValues(abacus,
		XtNvertical, state, NULL);
}

static void
stippleFrameCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#else
		Boolean state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif
	XtVaSetValues(abacus,
		XtNstippleFrame, state, NULL);
#ifdef LEE_ABACUS
	if (leftAuxAbacus)
		XtVaSetValues(leftAuxAbacus,
			XtNstippleFrame, state, NULL);
	if (rightAuxAbacus)
		XtVaSetValues(rightAuxAbacus,
			XtNstippleFrame, state, NULL);
#endif
}

static void
signCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#else
		Boolean state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif
	XtVaSetValues(abacus,
		XtNsign, state, NULL);
}

static void
subdecksSeparatedCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#else
		Boolean state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif
	XtVaSetValues(abacus,
		XtNsubdecksSeparated, state, NULL);
}

static void
altSubbeadPlacementCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#else
		Boolean state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif
	XtVaSetValues(abacus,
		XtNaltSubbeadPlacement, state, NULL);
}

static void
soundCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		Boolean state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif
	XtVaSetValues(abacus,
		XtNsound, state, NULL);
#ifdef LEE_ABACUS
	if (leftAuxAbacus)
		XtVaSetValues(leftAuxAbacus,
			XtNsound, state, NULL);
	if (rightAuxAbacus)
		XtVaSetValues(rightAuxAbacus,
			XtNsound, state, NULL);
#endif
}

static void
rightToLeftAddCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		int state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif
	XtVaSetValues(abacus,
		XtNrightToLeftAdd, state, NULL);
}

static void
rightToLeftMultCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		int state
#endif
		)
{
#ifdef HAVE_MOTIF
	Boolean state = cbs->set;
#endif

	XtVaSetValues(abacus,
		XtNrightToLeftMult, state, NULL);
}

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
railUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int rails;

	XtVaGetValues(abacus,
		XtNrails, &rails, NULL);
	rails++;
	setScale(railSlider, railSliderLabel, rails, MIN_RAILS, MAX_RAILS,
		False);
	XtVaSetValues(abacus,
		XtNrails, rails, NULL);
}

static void
railDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int rails;
	int min;
	int bottomPiece, bottomPiecePercent, shiftPercent;
	Boolean sign, demo;

	XtVaGetValues(abacus,
		XtNrails, &rails,
		XtNsign, &sign,
		XtNbottomPiece, &bottomPiece,
		XtNbottomPiecePercent, &bottomPiecePercent,
		XtNshiftPercent, &shiftPercent,
		XtNdemo, &demo, NULL);
	min = ((sign) ? 1 : 0) + ((bottomPiece == 0) ? 0 : 1) +
		((bottomPiecePercent == 0) ? 0 : 1 + shiftPercent) +
		((demo) ? MIN_DEMO_RAILS : MIN_RAILS);
	rails--;
	if (rails < min) {
		return;
	}
	setScale(railSlider, railSliderLabel, rails, MIN_RAILS, MAX_RAILS,
		False);
	XtVaSetValues(abacus,
		XtNrails, rails, NULL);
}

#elif defined(HAVE_MOTIF) || (defined(HAVE_ATHENA) && !defined(USE_SPIN))
static void
railChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int rails;
	int old;
	int bottomPiece, bottomPiecePercent, shiftPercent;
	Boolean sign, demo;

#ifdef HAVE_MOTIF
	int  min;
#ifdef USE_SPIN
	int limit;
	rails = cbs->position;
#else
	rails = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &rails,
			MIN_RAILS, MAX_RAILS, SCROLL_SIZE)) {
		return;
	}
#endif
	XtVaGetValues(abacus,
		XtNrails, &old,
		XtNsign, &sign,
		XtNbottomPiece, &bottomPiece,
		XtNbottomPiecePercent, &bottomPiecePercent,
		XtNshiftPercent, &shiftPercent,
		XtNdemo, &demo, NULL);
#ifdef HAVE_MOTIF
	min = ((sign) ? 1 : 0) + ((bottomPiece == 0) ? 0 : 1) +
		((bottomPiecePercent == 0) ? 0 : 1 + shiftPercent) +
		((demo) ? MIN_DEMO_RAILS : MIN_RAILS);
	if (rails < min) {
		setChanger(railSlider, old, spin);
		return;
	}
#endif
	if (old == rails)
		return;
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	XtVaGetValues(railSlider,
		XmNmaximumValue, &limit, NULL);
	if (rails >= limit)
		XtVaSetValues(railSlider,
			XmNmaximumValue, rails + 1,
			XmNposition, rails, NULL);
#else
	if (rails > MAX_RAILS)
		XtVaSetValues(railSlider,
			XmNmaximum, rails, NULL);
	XmScaleSetValue(railSlider, rails);
#endif
#elif defined(HAVE_ATHENA)
	setScale(w, label, rails, MIN_RAILS, MAX_RAILS, False);
#endif
	XtVaSetValues(abacus,
		XtNrails, rails, NULL);
}
#endif

static void
updateSpaceForBase(Boolean demo, int base)
{
	if (demo) {
		base = DEFAULT_BASE;
#ifdef HAVE_MOTIF
		setChanger(abacusBaseSlider, base, baseSpin);
#endif
	}
#if 0
/* It stopped making sense, had to do with 1/4,1/8,1/12 I think */
	/* Odd bases produce round-off errors but OK */
	/* When base divisible by 4, kind of silly but OK */
	/* Well some of these have enough room in Russian but not others */
	else if ((base == 2 || base == 4) && bottomSpaces < 3) {
		XtVaSetValues(abacus,
			XtNbottomSpaces, 3,
			XtNbase, base, NULL);
	} else if ((base == 3 || base == 6 || base == 9) && bottomSpaces < 2) {
		XtVaSetValues(abacus,
			XtNbottomSpaces, 2,
			XtNbase, base, NULL);
	} else {
		XtVaSetValues(abacus,
			XtNbase, base, NULL);
	}
#endif
}

static void
setDisplayBaseForLearning(Boolean learning, int base, int displayBase)
{
	XtVaSetValues(abacus,
		XtNdisplayBase, displayBase, NULL);
	if (learning) {
#ifdef HAVE_MOTIF
		setChanger(displayBaseSlider, base, baseSpin);
#endif
#ifdef LEE_ABACUS
		if (leftAuxAbacus)
			XtVaSetValues(leftAuxAbacus,
				XtNbase, base,
				XtNdisplayBase, displayBase, NULL);
		if (rightAuxAbacus)
			XtVaSetValues(rightAuxAbacus,
				XtNbase, base,
				XtNdisplayBase, displayBase, NULL);
#endif
	} else {
		XtVaSetValues(abacus,
			XtNdisplayBase, displayBase, NULL);
#ifdef LEE_ABACUS
		if (leftAuxAbacus)
			XtVaSetValues(leftAuxAbacus,
				XtNdisplayBase, displayBase, NULL);
		if (rightAuxAbacus)
			XtVaSetValues(rightAuxAbacus,
				XtNdisplayBase, displayBase, NULL);
#endif
	}
}

static void
setBaseForLearning(Boolean learning, int base)
{
	XtVaSetValues(abacus,
		XtNbase, base, NULL);
	if (learning) {
#ifdef HAVE_MOTIF
		setChanger(displayBaseSlider, base, baseSpin);
#endif
		XtVaSetValues(abacus,
			XtNdisplayBase, base, NULL);
#ifdef LEE_ABACUS
		if (leftAuxAbacus)
			XtVaSetValues(leftAuxAbacus,
				XtNbase, base,
				XtNdisplayBase, base, NULL);
		if (rightAuxAbacus)
			XtVaSetValues(rightAuxAbacus,
				XtNbase, base,
				XtNdisplayBase, base, NULL);
	} else {
		if (leftAuxAbacus)
			XtVaSetValues(leftAuxAbacus,
				XtNbase, base, NULL);
		if (rightAuxAbacus)
			XtVaSetValues(rightAuxAbacus,
				XtNbase, base, NULL);
#endif
	}
}

#if defined(HAVE_ATHENA) && defined(USE_BASE_SPIN)
static void
displayBaseUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int base, displayBase;
	Boolean demo, teach;

	XtVaGetValues(abacus,
		XtNbase, &base,
		XtNdisplayBase, &displayBase,
		XtNdemo, &demo,
		XtNteach, &teach, NULL);
	displayBase++;
	if (displayBase > MAX_BASE) {
		return;
	}
	if (demo)
		base = DEFAULT_BASE;
	setDisplayBaseForLearning((demo || teach), base, displayBase);
	setScale(displayBaseSlider, displayBaseSliderLabel, displayBase,
		MIN_BASE, MAX_BASE, True);
	XtVaSetValues(abacus,
		XtNdisplayBase, displayBase, NULL);
}

static void
displayBaseDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int base, displayBase;
	Boolean demo, teach;

	XtVaGetValues(abacus,
		XtNbase, &base,
		XtNdisplayBase, &displayBase,
		XtNdemo, &demo,
		XtNteach, &teach, NULL);
	displayBase--;
	if (displayBase < MIN_BASE) {
		return;
	}
	if (demo)
		base = DEFAULT_BASE;
	setDisplayBaseForLearning((demo || teach), base, displayBase);
	setScale(displayBaseSlider, displayBaseSliderLabel, displayBase,
		MIN_BASE, MAX_BASE, True);
	XtVaSetValues(abacus,
		XtNdisplayBase, displayBase, NULL);
}

static void
abacusBaseUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int base;
	Boolean demo, teach;

	XtVaGetValues(abacus,
		XtNbase, &base,
		XtNdemo, &demo,
		XtNteach, &teach, NULL);
	base++;
	if (base > MAX_BASE) {
		return;
	}
	updateSpaceForBase(demo, base);
	setBaseForLearning((demo || teach), base);
	setScale(abacusBaseSlider, abacusBaseSliderLabel, base,
		MIN_BASE, MAX_BASE, True);
	XtVaSetValues(abacus,
		XtNbase, base, NULL);
}

static void
abacusBaseDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int base;
	Boolean demo, teach;

	XtVaGetValues(abacus,
		XtNbase, &base,
		XtNdemo, &demo,
		XtNteach, &teach, NULL);
	base--;
	if (base < MIN_BASE) {
		return;
	}
	updateSpaceForBase(demo, base);
	setBaseForLearning((demo || teach), base);
	setScale(abacusBaseSlider, abacusBaseSliderLabel, base,
		MIN_BASE, MAX_BASE, True);
	XtVaSetValues(abacus,
		XtNbase, base, NULL);
}

#elif defined(HAVE_MOTIF) || (defined(HAVE_ATHENA) && !defined(USE_BASE_SPIN))
static void
displayBaseChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_BASE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int displayBase, old, base;
	Boolean demo, teach;

#ifdef HAVE_MOTIF
#ifdef USE_BASE_SPIN
	displayBase = cbs->position;
#else
	displayBase = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &displayBase, MIN_BASE, MAX_BASE,
			2 * SCROLL_SIZE)) {
		return;
	}
#endif
	XtVaGetValues(abacus,
		XtNbase, &base,
		XtNdisplayBase, &old,
		XtNdemo, &demo,
		XtNteach, &teach, NULL);
	if (old == displayBase)
		return;
	if (demo)
		base = DEFAULT_BASE;
	setDisplayBaseForLearning((demo || teach), base, displayBase);
#ifdef HAVE_ATHENA
	setScale(w, label, displayBase, MIN_BASE, MAX_BASE, True);
#endif
}

static void
abacusBaseChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_BASE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int base, old;
	Boolean demo, teach;

#ifdef HAVE_MOTIF
#ifdef USE_BASE_SPIN
	base = cbs->position;
#else
	base = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &base, MIN_BASE, MAX_BASE,
			2 * SCROLL_SIZE)) {
		return;
	}
#endif
	XtVaGetValues(abacus,
		XtNbase, &old,
		XtNdemo, &demo,
		XtNteach, &teach, NULL);
	if (old == base)
		return;
	updateSpaceForBase(demo, base);
	setBaseForLearning((demo || teach), base);
#ifdef HAVE_ATHENA
	setScale(w, label, base, MIN_BASE, MAX_BASE, True);
#endif
}
#endif

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
delayUpCallback(Widget w, XtPointer clientData, XtPointer callData) {
	int delay;

	XtVaGetValues(abacus,
		XtNdelay, &delay, NULL);
	delay++;
	setScale(delaySlider, delaySliderLabel, delay, 0, MAX_DELAY, False);
	XtVaSetValues(abacus,
		XtNdelay, delay, NULL);
}

static void
delayDownCallback(Widget w, XtPointer clientData, XtPointer callData) {
	int delay;

	XtVaGetValues(abacus,
		XtNdelay, &delay, NULL);
	delay--;
	if (delay < 0) {
		return;
	}
	setScale(delaySlider, delaySliderLabel, delay, 0, MAX_DELAY, False);
	XtVaSetValues(abacus,
		XtNdelay, delay, NULL);
}

#elif defined(HAVE_MOTIF) || (defined(HAVE_ATHENA) && !defined(USE_SPIN))
static void
delayChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int delay, old; /* min? */

#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	delay = cbs->position;
#else
	delay = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &delay, 0, MAX_DELAY,
			SCROLL_SIZE)) {
		return;
	}
#endif
	XtVaGetValues(abacus,
		XtNdelay, &old, NULL);
	if (old == delay)
		return;
#ifdef HAVE_ATHENA
	setScale(w, label, delay, 0, MAX_DELAY, False);
#endif
	XtVaSetValues(abacus,
		XtNdelay, delay, NULL);
}
#endif

static void
formatCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	Boolean demo, slot;
	int subbase, old;
	int mode = (size_t) clientData;

	XtVaGetValues(abacus,
		XtNmode, &old,
		XtNslot, &slot,
		XtNsubbase, &subbase,
		XtNdemo, &demo, NULL);
	if (mode < 0 || mode > MAX_FORMATS)
		mode = GENERIC;
	if (old == mode)
		return;
#ifdef HAVE_MOTIF
	if (demo && mode == GENERIC) {
		XtVaSetValues(formatMenu,
			XmNmenuHistory, formatOptions[old], NULL);
		return;
	}
#elif defined(HAVE_ATHENA)
	XtVaSetValues(formatLabel,
		XtNlabel, formatChoices[mode], NULL);
#endif
	checkEnabled(abacus);
	if (old == ROMAN && mode != ROMAN && subbase) {
		XtVaSetValues(abacus,
			XtNdeck, CLEAR_DECIMAL_DECK, NULL);
	}
	if (mode != GENERIC)
		slot = (mode == ROMAN); /* since just changed */
	XtVaSetValues(abacus,
		XtNmode, mode,
		XtNslot, slot, NULL);
	if (demo) {
		XtVaSetValues(abacusDemo,
			XtNdeck, CLEAR_DECK,
			XtNmode, mode, NULL);
	}
}

static void
romanNumeralsMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	int old;

	XtVaGetValues(abacus,
		XtNromanNumeralsMode, &old, NULL);
	if (old == value)
		return;
#ifdef HAVE_ATHENA
	XtVaSetValues(romanNumeralsLabel,
		XtNlabel, romanFormatChoices[value], NULL);
#endif
	XtVaSetValues(abacus,
		XtNromanNumeralsMode, value, NULL);
}

static void
romanMarkersMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	int old;

	XtVaGetValues(abacus,
		XtNromanMarkersMode, &old, NULL);
	if (old == value)
		return;
#ifdef HAVE_ATHENA
	XtVaSetValues(romanMarkersLabel,
		XtNlabel, romanFormatChoices[value], NULL);
#endif
	XtVaSetValues(abacus,
		XtNromanMarkersMode, value, NULL);
	checkEnabled(abacus);
}

static void
pieceRailMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	int old, top, bottom;

	XtVaGetValues(abacus,
		XtNtopPiece, &top,
		XtNbottomPiece, &bottom, NULL);
	old = PIECE_CHOICE(top, bottom);
	if (old == value)
		return;
#ifdef HAVE_ATHENA
	XtVaSetValues(pieceRailLabel,
		XtNlabel, pieceChoices[value], NULL);
#endif
	if (old > 0 && value > 0) /* reset */
		XtVaSetValues(abacus,
			XtNtopPiece, 0,
			XtNbottomPiece, 0, NULL);
	top = PIECE_TOP(value);
	bottom = PIECE_BOTTOM(value);
	XtVaSetValues(abacus,
		XtNtopPiece, top,
		XtNbottomPiece, bottom, NULL);
	checkEnabled(abacus);
}

static void
piecePercentRailMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	int old, top, bottom;

	XtVaGetValues(abacus,
		XtNtopPiecePercent, &top,
		XtNbottomPiecePercent, &bottom, NULL);
	old = PIECE_CHOICE(top, bottom);
	if (old == value)
		return;
#ifdef HAVE_ATHENA
	XtVaSetValues(piecePercentRailLabel,
		XtNlabel, pieceChoices[value], NULL);
#endif
	if (old > 0 && value > 0)
		XtVaSetValues(abacus,
			XtNtopPiecePercent, 0,
			XtNbottomPiecePercent, 0, NULL);
	top = PIECE_TOP(value);
	bottom = PIECE_BOTTOM(value);
	XtVaSetValues(abacus,
		XtNtopPiecePercent, top,
		XtNbottomPiecePercent, bottom, NULL);
	checkEnabled(abacus);
}

static void
subdeckRailMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	int old, subbase;

	XtVaGetValues(abacus,
		XtNsubbase, &old, NULL);
	if (old == value)
		return;
#ifdef HAVE_ATHENA
	XtVaSetValues(subdeckRailLabel,
		XtNlabel, pieceChoices[value], NULL);
#endif
	subbase = PIECE_BASE(value);
	XtVaSetValues(abacus,
		XtNsubbase, subbase, NULL);
	checkEnabled(abacus);
}

static void
museumMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	int old;

	XtVaGetValues(abacus,
		XtNmuseumMode, &old, NULL);
	if (old == value)
		return;
#ifdef HAVE_ATHENA
	XtVaSetValues(museumLabel,
		XtNlabel, museumChoices[value], NULL);
#endif
	XtVaSetValues(abacus,
		XtNmuseumMode, value, NULL);
}

#ifdef ANOMALY
static void
anomalyMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	int old, anomaly, anomalySq;

	XtVaGetValues(abacus,
		XtNanomaly, &anomaly,
		XtNanomalySq, &anomalySq, NULL);
	old = ANOMALY_CHOICE(anomaly, anomalySq);
	if (old == value)
		return;
#ifdef HAVE_ATHENA
	XtVaSetValues(anomalyLabel,
		XtNlabel, anomalyChoices[value], NULL);
#endif
	anomaly = ANOMALY_P(value);
	anomalySq = ANOMALY_SQ(value);
	XtVaSetValues(abacus,
		XtNanomaly, anomaly,
		XtNanomalySq, anomalySq, NULL);
	checkEnabled(abacus);
}
#endif

static void
abacusClearCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmAnyCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
#ifdef HAVE_MOTIF
	if (cbs->reason != XmCR_OK)
		return;
#elif defined(HAVE_ATHENA)
	XtPopdown(clearDialog);
#endif
	XtVaSetValues(abacus,
		XtNmenu, ACTION_CLEAR, NULL);
#ifdef LEE_ABACUS
	if (leftAuxAbacus)
		XtVaSetValues(leftAuxAbacus,
			XtNmenu, ACTION_CLEAR, NULL);
	if (rightAuxAbacus)
		XtVaSetValues(rightAuxAbacus,
			XtNmenu, ACTION_CLEAR, NULL);
#endif
}

static void
fileMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);

	if (value == 0)
		exit(0);
}

#ifdef HAVE_ATHENA
static void
controlsMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);

	if (value < 4)
		XtVaSetValues(abacus,
			XtNmenu, value + ACTION_CLEAR, NULL);
	switch (value) {
	case 4:
		createDialog(topLevel, displayDialogShell);
		return;
	case 5:
		createDialog(topLevel, specialRailsDialogShell);
		return;
	case 6:
		createDialog(topLevel, secondaryRailsDialogShell);
		return;
	case 7:
		createDialog(topLevel, setupDialogShell);
		return;
	}
}
#endif

static void
learnMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	Boolean wasDemo, wasTeach;

#ifdef HAVE_ATHENA
	if (value == 0) {
		createDialog(topLevel, calculateDialogShell);
		return;
	} else
		value--;
#endif
	XtVaGetValues(abacus,
		XtNdemo, &wasDemo,
		XtNteach, &wasTeach, NULL);
	if (wasDemo) {
		XtVaSetValues(abacus,
			XtNdemo, !wasDemo, NULL);
		XtDestroyWidget(abacusDemo);
#ifdef HAVE_MOTIF
		XtVaSetValues(demoToggle,
			XmNset, False, NULL);
		XtVaSetValues(teachToggle,
			XmNset, False, NULL);
#elif defined(HAVE_ATHENA)
		setDemoCheck(demoMenuItem);
#endif
		return;
	}
	if (wasTeach) {
		XtVaSetValues(abacus,
			XtNteach, !wasTeach, NULL);
#ifdef HAVE_MOTIF
		destroyTeach();
		XtVaSetValues(demoToggle,
			XmNset, False, NULL);
		XtVaSetValues(teachToggle,
			XmNset, False, NULL);
#elif defined(HAVE_ATHENA)
		XtPopdown(teachDialogShell);
#endif
		return;
	}
	if (value == 0) {
		forceDemoParams(abacus);
		XtVaSetValues(abacus,
			XtNdemo, !wasDemo, NULL);
		createDemo();
		realizeDemo();
#ifdef HAVE_ATHENA
		setDemoCheck(demoMenuItem);
#endif
	} else if (value == 1) {
		XtVaSetValues(abacus,
			XtNteach, !wasTeach, NULL);
		forceTeachParams(abacus);
#ifdef HAVE_MOTIF
		createTeach(abacus);
#elif defined(HAVE_ATHENA)
		createDialog(topLevel, teachDialogShell);
		initializeText((AbacusWidget) abacus);
#endif
	}
}

static void
helpMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	Widget dialog;

#ifdef HAVE_ATHENA
	if (wmDeleteWindow == None)
		wmDeleteWindow = XInternAtom(XtDisplay(topLevel),
			"WM_DELETE_WINDOW", FALSE);
#endif
	switch (value) {
	case 0:
		dialog = descriptionDialog;
		break;
	case 1:
		dialog = featuresDialog;
		break;
	case 2:
		dialog = synopsisDialog;
		break;
	case 3:
		dialog = optionsDialog;
		break;
	case 4:
		dialog = referencesDialog;
		break;
	case 5:
		dialog = aboutDialog;
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpMenuCallback: %d", value);
			XtWarning(buf);
			free(buf);
			return;
		}
	}
#ifdef HAVE_MOTIF
	XtManageChild(dialog);
#elif defined(HAVE_ATHENA)
	XtPopup(dialog, XtGrabNone);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(dialog), &wmDeleteWindow, 1);
#endif
}
#endif

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static void
setRomanNumeralsFormat(int mode)
{
#ifdef HAVE_MOTIF
	setRadio(romanNumeralsMenu, mode, ROMAN_NUMERALS_FORMATS);
#elif defined(HAVE_ATHENA)
	setRadio(romanNumeralsLabel, romanFormatChoices[mode]);
#endif
}

static void
setRomanMarkersFormat(int mode)
{
#ifdef HAVE_MOTIF
	setRadio(romanMarkersMenu, mode, ROMAN_MARKERS_FORMATS);
#elif defined(HAVE_ATHENA)
	setRadio(romanMarkersLabel, romanFormatChoices[mode]);
#endif
}

static void
setPiece(int mode)
{
#ifdef HAVE_MOTIF
	setRadio(pieceRailMenu, mode, PIECE_OPTIONS);
#elif defined(HAVE_ATHENA)
	setRadio(pieceRailLabel, pieceChoices[mode]);
#endif
}

static void
setPiecePercent(int mode)
{
#ifdef HAVE_MOTIF
	setRadio(piecePercentRailMenu, mode, PIECE_OPTIONS);
#elif defined(HAVE_ATHENA)
	setRadio(piecePercentRailLabel, pieceChoices[mode]);
#endif
}

static void
setSubdeck(int mode)
{
#ifdef HAVE_MOTIF
	setRadio(subdeckRailMenu, mode, PIECE_OPTIONS);
#elif defined(HAVE_ATHENA)
	setRadio(subdeckRailLabel, pieceChoices[mode]);
#endif
}

static void
setMuseum(int mode)
{
#ifdef HAVE_MOTIF
	setRadio(museumMenu, mode, MAX_MUSEUMS);
#elif defined(HAVE_ATHENA)
	setRadio(museumLabel, museumChoices[mode]);
#endif
}

#ifdef ANOMALY
static void
setAnomaly(int mode)
{
#ifdef HAVE_MOTIF
	setRadio(anomalyMenu, mode, ANOMALY_OPTIONS);
#elif defined(HAVE_ATHENA)
	setRadio(anomalyLabel, anomalyChoices[mode]);
#endif
}
#endif
#endif

static void
abacusCallback(Widget w, caddr_t clientData,
		abacusCallbackStruct *callData)
{
	int rails, mode;
	Boolean subdecksSeparated, altSubbeadPlacement;
	Boolean sign, demo, teach;
	Boolean latin, group, placeOnRail, vertical;
	int bottomPiece;
	int topPiece, topPiecePercent, bottomPiecePercent;
#ifdef ANOMALY
	int anomaly, anomalySq;
#endif
	Boolean rightToLeftAdd, rightToLeftMult;
	int subbase, museumMode, romanNumeralsMode, romanMarkersMode;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
#if defined(HAVE_MOTIF) && defined(USE_SPIN)
	int limit;
#elif defined(HAVE_ATHENA)
	int max;
	char *defaultString;
#endif
	Boolean stippleFrame, sound;
#endif

	XtVaGetValues(w,
		XtNdemo, &demo,
		XtNteach, &teach,
		XtNmode, &mode, NULL);
	switch (callData->reason) {
	case ACTION_HIDE:
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		if (demo)
			(void) XIconifyWindow(XtDisplay(shell),
				XtWindow(shell),
				XScreenNumberOfScreen(XtScreen(shell)));
#endif
		break;
#ifdef CLEAR_QUERY
	case ACTION_CLEAR_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(clearDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(clearDialog, XtGrabNone);
		XSetWMProtocols(XtDisplay(topLevel),
			XtWindow(clearDialog), &wmDeleteWindow, 1);
#else
		XtVaSetValues(w
			XtNmenu, ACTION_CLEAR, NULL);
#endif
		break;
#endif
	case ACTION_PAUSE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(pauseDialog);
#ifdef LEE_ABACUS
		if (leftAuxTracker) {
			if (mathBuffer)
				free(mathBuffer);
			mathBuffer = XmTextGetString(leftAuxTracker);
			printState(leftAuxTracker, mathBuffer);
		}
		if (rightAuxTracker) {
			if (mathBuffer)
				free(mathBuffer);
			mathBuffer = XmTextGetString(rightAuxTracker);
			printState(rightAuxTracker, mathBuffer);
		}
#endif
		if (mathBuffer)
			free(mathBuffer);
		mathBuffer = XmTextGetString(tracker);
		printState(tracker, mathBuffer);
#endif
		break;
	case ACTION_SCRIPT:
		if (callData->deck != 0 || callData->number != 0) {
			(void) printf("  <move>\n");
			(void) printf("    <code>%d %d %d %d 4</code>\n",
				callData->aux, callData->deck,
				callData->rail, callData->number);
			(void) printf("    <text>\n");
			(void) printf("      Lesson\n\n\n");
			(void) printf("      Press Space-bar to Continue\n");
			(void) printf("    </text>\n");
			(void) printf("  </move>\n");
		}
		break;
	case ACTION_MOVE:
#if 0
		(void) printf("xabacus MOVE: %d %d %d %d\n",
			callData->aux, callData->deck, callData->rail,
			callData->number);
#endif
#ifndef LEE_ABACUS
		if (callData->aux == PRIMARY)
#endif
		{
			XtVaSetValues(AUXWIN(callData->aux),
				XtNdeck, callData->deck,
				XtNrail, callData->rail,
				XtNnumber, callData->number, NULL);
		}
		break;
	case ACTION_CLEAR:
		if (callData->aux == PRIMARY)
			XtVaSetValues(abacus,
				XtNdeck, CLEAR_DECK, NULL);
#ifdef LEE_ABACUS
		else if (callData->aux == LEFT_AUX && leftAuxAbacus)
			XtVaSetValues(leftAuxAbacus,
				XtNdeck, CLEAR_DECK, NULL);
		else if (callData->aux == RIGHT_AUX && rightAuxAbacus)
			XtVaSetValues(rightAuxAbacus,
				XtNdeck, CLEAR_DECK, NULL);
#endif
		else {
			XtVaSetValues(abacus,
				XtNdeck, CLEAR_DECK, NULL);
#ifdef LEE_ABACUS
			if (leftAuxAbacus)
				XtVaSetValues(leftAuxAbacus,
					XtNdeck, CLEAR_DECK, NULL);
			if (rightAuxAbacus)
				XtVaSetValues(rightAuxAbacus,
					XtNdeck, CLEAR_DECK, NULL);
#endif
		}
		break;
	case ACTION_DEMO:
		demo = !demo;
		if (demo) {
			forceDemoParams(abacus);
			createDemo();
			realizeDemo();
		} else {
			XtDestroyWidget(abacusDemo);
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
			/* Destroying shell is causing trouble, so lets not do that */
			/* Warning: XtRemoveGrab asked to remove a widget not on the list */
			/* http://www.mail-archive.com/lesstif@hungry.com/msg00535.html */
			/*XtDestroyWidget(shell);
			shell = NULL;*/
			XtUnrealizeWidget(shell);
#endif
		}
		XtVaSetValues(abacus,
			XtNdemo, demo, NULL);
#ifdef LEE_ABACUS
		if (leftAuxAbacus)
			XtVaSetValues(leftAuxAbacus,
				XtNdemo, demo, NULL);
		if (rightAuxAbacus)
			XtVaSetValues(rightAuxAbacus,
				XtNdemo, demo, NULL);
#endif
#ifdef HAVE_MOTIF
		XtVaSetValues(demoToggle,
			XmNset, demo, NULL);
		/*XtVaSetValues(choiceMenu,
			XmNmenuHistory, choiceOptions[(demo) ? DEMO : NORMAL], NULL);*/
#elif defined(HAVE_ATHENA)
		setDemoCheck(demoMenuItem);
#endif
		break;
#ifdef HAVE_MOTIF
	case ACTION_TEACH:
		if (teachTracker == NULL) {
			forceTeachParams(abacus);
			createTeach(abacus);
		} else {
			destroyTeach();
		}
		XtVaSetValues(abacus, XtNteach,
			(teachTracker == NULL) ? False : True, NULL);
		XtVaSetValues(teachToggle,
			XmNset, (teachTracker == NULL) ? False : True, NULL);
		/*XtVaSetValues(choiceMenu,
			XmNmenuHistory, choiceOptions[(demo) ? TEACH : NORMAL], NULL);*/
		break;
	case ACTION_TEACH_LINE:
		if (callData->line >= 0 && callData->line <= 2) {
			/* Tried a few things to get accented chars to work directly */
			/*static XmStringCharSet charSet =
				(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;*/
			/*static XmStringCharSet charSet = "UTF-8";*/
			XmString teachBuffer = NULL;

			if (teachTracker == NULL)
				createTeach(abacus);
			/*teachBuffer = XmStringCreate(callData->teachBuffer, charSet);*/
			/*teachBuffer = XmStringCreateLtoR(callData->teachBuffer, charSet);*/
			teachBuffer = XmStringCreateLocalized(callData->teachBuffer);
			/*teachBuffer = XmStringGenerate(callData->teachBuffer, NULL, XmWIDECHAR_TEXT, NULL);*/
			XtVaSetValues(teachLine[callData->line],
				XmNlabelString, teachBuffer, NULL);
			XmStringFree(teachBuffer);
		}
		break;
#elif defined(HAVE_ATHENA)
	case ACTION_TEACH:
		teach = !teach;
		if (teach) {
			createDialog(topLevel, teachDialogShell);
			initializeText((AbacusWidget) abacus);
		} else {
			XtPopdown(teachDialogShell);
		}
		XtVaSetValues(abacus,
			XtNteach, teach, NULL);
		break;
	case ACTION_TEACH_LINE:
		if (callData->line >= 0 && callData->line <= 2) {
			XtVaSetValues(teachLine[callData->line],
				XtNlabel, callData->teachBuffer, NULL);
		}
		break;
#endif
	case ACTION_NEXT:
		XtVaSetValues(abacusDemo, XtNdeck, NEXT_DECK, NULL);
		break;
	case ACTION_ENTER:
		XtVaSetValues(abacusDemo, XtNdeck, ENTER_DECK, NULL);
		break;
	case ACTION_JUMP:
		XtVaSetValues(abacusDemo, XtNdeck, JUMP_DECK, NULL);
		break;
	case ACTION_MORE:
		XtVaSetValues(abacusDemo, XtNdeck, MORE_DECK, NULL);
		break;
	case ACTION_PLACE:
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			checkEnabled(w);
		}
#endif
		break;
	case ACTION_INCREMENT:
		if (w == abacus) {
			XtVaGetValues(w,
				XtNrails, &rails, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
			XtVaGetValues(railSlider,
				XmNmaximumValue, &limit, NULL);
			if (rails >= limit)
				XtVaSetValues(railSlider,
					XmNposition, rails,
					XmNmaximumValue, rails + 1, NULL);
			else
				XtVaSetValues(railSlider,
					XmNposition, rails, NULL);
#else
			if (rails > MAX_RAILS)
				XtVaSetValues(railSlider,
					XmNmaximum, rails, NULL);
			XmScaleSetValue(railSlider, rails);
#endif
#elif defined(HAVE_ATHENA)
			setScale(railSlider, railSliderLabel, rails,
				MIN_RAILS, MAX_RAILS, True);
#endif
		}
		break;
	case ACTION_DECREMENT:
		if (w == abacus) {
			XtVaGetValues(w,
				XtNrails, &rails, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
			if (rails > MAX_RAILS)
				XtVaSetValues(railSlider,
					XmNmaximumValue, rails,
					XmNposition, rails, NULL);
			else
				XtVaSetValues(railSlider,
					XmNposition, rails, NULL);
#else
			if (rails >= MAX_RAILS)
				XtVaSetValues(railSlider,
					XmNmaximum, rails, NULL);
			XmScaleSetValue(railSlider, rails);
#endif
#elif defined(HAVE_ATHENA)
			setScale(railSlider, railSliderLabel, rails,
				MIN_RAILS, MAX_RAILS, True);
#endif
		}
		break;
	case ACTION_FORMAT:
		if (w == abacus) {
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
#ifdef HAVE_MOTIF
			XtVaSetValues(formatMenu,
				XmNmenuHistory, formatOptions[mode], NULL);
			if (demo) {
				XtVaSetValues(abacusDemo,
					XtNdeck, CLEAR_DECK,
					XtNmode, mode, NULL);
			}
#else
			max = findMaxLength((char **) formatChoices,
				sizeof(formatChoices) / sizeof(*formatChoices));
			createBlank(&defaultString, max, (char *) formatChoices[mode], 0);
			XtVaSetValues(formatLabel,
				XtNlabel, defaultString, NULL);
			free(defaultString);
#endif
			checkEnabled(w);
#endif
		}
		break;
	case ACTION_ROMAN_NUMERALS:
		XtVaGetValues(w,
			XtNromanNumeralsMode, &romanNumeralsMode,
			XtNbottomPiece, &bottomPiece, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setRomanNumeralsFormat(romanNumeralsMode);
			XtSetSensitive(latinToggle,
				ENABLE_LATIN(romanNumeralsMode, bottomPiece));
		}
#endif
		break;
	case ACTION_ROMAN_MARKERS:
		XtVaGetValues(w,
			XtNromanMarkersMode, &romanMarkersMode, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setRomanMarkersFormat(romanMarkersMode);
			checkEnabled(w);
		}
#endif
		break;
	case ACTION_LATIN:
		XtVaGetValues(w,
			XtNlatin, &latin, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setToggle(latinToggle, latin, False);
		}
#endif
		break;
	case ACTION_GROUP:
		XtVaGetValues(w,
			XtNgroup, &group, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setToggle(groupToggle, group, False);
		}
#endif
		break;
	case ACTION_PLACE_ON_RAIL:
		XtVaGetValues(w,
			XtNplaceOnRail, &placeOnRail, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setToggle(placeOnRailToggle, placeOnRail, False);
		}
#endif
		break;
	case ACTION_VERTICAL:
		XtVaGetValues(w,
			XtNvertical, &vertical, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setToggle(verticalToggle, vertical, False);
		}
#endif
		break;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	case ACTION_STIPPLE_FRAME:
		XtVaGetValues(w,
			XtNstippleFrame, &stippleFrame, NULL);
#ifdef LEE_ABACUS
		XtVaSetValues(abacus,
			XtNstippleFrame, stippleFrame, NULL);
		if (leftAuxAbacus)
			XtVaSetValues(leftAuxAbacus,
				XtNstippleFrame, stippleFrame, NULL);
		if (rightAuxAbacus)
			XtVaSetValues(rightAuxAbacus,
				XtNstippleFrame, stippleFrame, NULL);
#endif
		setToggle(stippleFrameToggle, stippleFrame, False);
		break;
#endif
	case ACTION_SIGN:
		XtVaGetValues(w,
			XtNsign, &sign, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setToggle(signToggle, sign, False);
		}
#endif
		break;
	case ACTION_PIECE:
		XtVaGetValues(w,
			XtNtopPiece, &topPiece,
			XtNbottomPiece, &bottomPiece, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setPiece(PIECE_CHOICE(topPiece, bottomPiece));
			checkEnabled(w);
		}
#endif
		break;
	case ACTION_PIECE_PERCENT:
		XtVaGetValues(w,
			XtNtopPiecePercent, &topPiecePercent,
			XtNbottomPiecePercent, &bottomPiecePercent, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setPiecePercent(PIECE_CHOICE(topPiecePercent,
				bottomPiecePercent));
			checkEnabled(w);
		}
#endif
		break;
	case ACTION_SUBDECK:
		XtVaGetValues(w,
			XtNsubbase, &subbase, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setSubdeck(subbase >> 2);
			checkEnabled(w);
		}
#endif
		break;
	case ACTION_SUBDECKS_SEPARATED:
		XtVaGetValues(w,
			XtNsubdecksSeparated, &subdecksSeparated, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setToggle(subdecksSeparatedToggle, subdecksSeparated, False);
		}
#endif
		break;
	case ACTION_ALT_SUBBEAD_PLACEMENT:
		XtVaGetValues(w,
			XtNaltSubbeadPlacement, &altSubbeadPlacement, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setToggle(altSubbeadPlacementToggle, altSubbeadPlacement, False);
		}
#endif
		break;
	case ACTION_MUSEUM:
		XtVaGetValues(w,
			XtNmuseumMode, &museumMode, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setMuseum(museumMode);
		}
#endif
		break;
#ifdef ANOMALY
	case ACTION_ANOMALY:
		XtVaGetValues(w,
			XtNanomaly, &anomaly,
			XtNanomalySq, &anomalySq,
			XtNromanMarkersMode, &romanMarkersMode, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setAnomaly(ANOMALY_CHOICE(anomaly, anomalySq));
#ifdef HAVE_MOTIF
			XtSetSensitive(romanMarkersMenu,
				ENABLE_ROMAN_MARKERS(anomaly, anomalySq));
#elif defined(HAVE_ATHENA)
			XtSetSensitive(romanMarkersLabel,
				ENABLE_ROMAN_MARKERS(anomaly, anomalySq));
#endif
		}
#endif
		break;
#endif
	case ACTION_RIGHT_TO_LEFT_ADD :
		XtVaGetValues(w,
			XtNrightToLeftAdd, &rightToLeftAdd, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setToggle(rightToLeftAddToggle, rightToLeftAdd, False);
		}
#endif
		break;
	case ACTION_RIGHT_TO_LEFT_MULT:
		XtVaGetValues(w,
			XtNrightToLeftMult, &rightToLeftMult, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		if (w == abacus) {
			setToggle(rightToLeftMultToggle, rightToLeftMult, False);
		}
#endif
		break;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	case ACTION_SOUND:
		XtVaGetValues(w,
			XtNsound, &sound, NULL);
#ifdef LEE_ABACUS
		XtVaSetValues(abacus,
			XtNsound, sound, NULL);
		if (leftAuxAbacus)
			XtVaSetValues(leftAuxAbacus,
				XtNsound, sound, NULL);
		if (rightAuxAbacus)
			XtVaSetValues(rightAuxAbacus,
				XtNsound, sound, NULL);
#endif
		setToggle(soundToggle, sound, False);
		break;
#endif
	}
	if (/*callData->reason == ACTION_SCRIPT || */callData->reason == ACTION_IGNORE) {
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
#ifdef LEE_ABACUS
		int initialValue = False;

		if (strcmp(callData->buffer, "0.0") == 0 ||
				strcmp(callData->buffer, "0") == 0) {
			initialValue = True;
		}
		if (leftAuxTracker && leftAuxAbacus && w == leftAuxAbacus) {
			static Pixel leftAuxOrigColor;
			static Pixel leftAuxBeadColor;
			static int leftAuxFirstTime = True;
			static int leftAuxInitialColor = True;

			if (leftAuxFirstTime) {
				XtVaGetValues(w,
					XtNforeground, &leftAuxOrigColor,
					XtNleftAuxBeadColor, &leftAuxBeadColor, NULL);
				leftAuxBeadColor = darker((AbacusWidget) leftAuxAbacus,
					darker((AbacusWidget) leftAuxAbacus, leftAuxBeadColor));
				leftAuxFirstTime = False;
			}
			if (initialValue && leftAuxInitialColor) {
				XtVaSetValues(leftAuxTracker,
					XtNforeground, leftAuxBeadColor, NULL);
				printState(leftAuxTracker, (char *) "Left Auxiliary");
			} else {
				XtVaSetValues(leftAuxTracker,
					XtNforeground, leftAuxOrigColor, NULL);
				printState(leftAuxTracker, callData->buffer);
				leftAuxInitialColor = False;
			}
		} else if (rightAuxTracker && rightAuxAbacus && w == rightAuxAbacus) {
			static Pixel rightAuxOrigColor;
			static Pixel rightAuxBeadColor;
			static int rightAuxFirstTime = True;
			static int rightAuxInitialColor = True;

			if (rightAuxFirstTime) {
				XtVaGetValues(w,
					XtNforeground, &rightAuxOrigColor,
					XtNrightAuxBeadColor, &rightAuxBeadColor, NULL);
				rightAuxBeadColor = darker((AbacusWidget) rightAuxAbacus,
					darker((AbacusWidget) rightAuxAbacus, rightAuxBeadColor));
				rightAuxFirstTime = False;
			}
			if (initialValue && rightAuxInitialColor) {
				XtVaSetValues(rightAuxTracker,
					XtNforeground, rightAuxBeadColor, NULL);
				printState(rightAuxTracker, (char *) "Right Auxiliary");
			} else {
				XtVaSetValues(rightAuxTracker,
					XtNforeground, rightAuxOrigColor, NULL);
				printState(rightAuxTracker, callData->buffer);
				rightAuxInitialColor = False;
			}
		} else if (leftAuxAbacus || rightAuxAbacus) {
			static Pixel primaryOrigColor;
			static Pixel primaryBeadColor;
			static int primaryFirstTime = True;
			static int primaryInitialColor = True;

			if (primaryFirstTime) {
				XtVaGetValues(w,
					XtNforeground, &primaryOrigColor,
					XtNprimaryBeadColor, &primaryBeadColor, NULL);
				primaryBeadColor = darker((AbacusWidget) abacus,
					darker((AbacusWidget) abacus, primaryBeadColor));
				primaryFirstTime = False;
			}
			if (initialValue && primaryInitialColor) {
				XtVaSetValues(tracker,
					XtNforeground, primaryBeadColor, NULL);
				printState(tracker, (char *) "Primary");
			} else {
				XtVaSetValues(tracker,
					XtNforeground, primaryOrigColor, NULL);
				printState(tracker, callData->buffer);
				primaryInitialColor = False;
			}
		} else {
			printState(tracker, callData->buffer);
		}
#else
		printState(tracker, callData->buffer);
#endif
#elif defined(HAVE_ATHENA)
		printState(tracker, callData->buffer);
#else
		printState(XtParent(w), callData->buffer,
			((mode < 0 || mode >= MAX_FORMATS)
			? "Abacus" : formatChoices[mode]));
#endif
	}
}

#ifdef HAVE_MOTIF
static void
abacusMathCallback(Widget w, caddr_t clientData, abacusCallbackStruct *callData)
{
	Boolean demo, teach;
	unsigned int i, j = 0;
	int aux;

	XtVaGetValues(abacus,
		XtNdemo, &demo,
		XtNteach, &teach, NULL);
	if (demo || teach)
		return;
#ifdef LEE_ABACUS
	if (leftAuxTracker && leftAuxAbacus && w == leftAuxTracker) {
		aux = LEFT_AUX;
	} else if (rightAuxTracker && rightAuxAbacus &&
			w == rightAuxTracker) {
		aux = RIGHT_AUX;
	} else
#endif
	{
		aux = PRIMARY;
	}
	if (mathBuffer)
		free(mathBuffer);
#ifdef HAVE_MOTIF
	mathBuffer = XmTextGetString(w);
#elif defined(HAVE_ATHENA)
	/*mathBuffer = XawDialogGetValueString(w);*/
	XtVaGetValues(w,
		XtNstring, &mathBuffer, NULL);
#endif
	/* strip out blanks */
	for (i = 0; i < strlen(mathBuffer); i++) {
		if (mathBuffer[i] == '[' || mathBuffer[i] == ']') {
			mathBuffer[j] = '\0';
			break;
		} else if (mathBuffer[i] != ' ' &&
				mathBuffer[i] != '\t') {
			mathBuffer[j] = mathBuffer[i];
			j++;
		}
	}
	/* check for math ops */
	XtVaSetValues(abacus,
		XtNaux, aux,
		XtNdeck, CALC_DECK,
		XtNmathBuffer, mathBuffer, NULL);
}
#endif

#ifdef HAVE_MOTIF
static void
abacusPauseCallback(Widget w, XtPointer clientData, XmAnyCallbackStruct *cbs)
{
}

static void
nullCallback(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct *cbs)
{
}

static Widget
createBaseChangers(Widget w, char *title, Boolean spin)
{
	Widget button, dialog, pane;
	Widget changerRowCol;
	Widget changerDisplayRowCol;
	Widget changerAbacusRowCol;
	int base, displayBase;
	char titleDsp[TITLE_FILE_LENGTH];
	XmString titleString = NULL;

	concatTitle(titleDsp, progDsp, ": ", title, TITLE_FILE_LENGTH);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(args[0], XmNdialogTitle, titleString);
	dialog = XmCreateMessageDialog(w, title, args, 1);
	XmStringFree(titleString);
	pane = XtVaCreateWidget("pane", xmPanedWindowWidgetClass, dialog,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	changerRowCol = XtVaCreateManagedWidget("changerRowCol",
		xmRowColumnWidgetClass, pane,
		XmNnumColumns, 1,
		XmNorientation, XmVERTICAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	changerDisplayRowCol = XtVaCreateManagedWidget("changerDisplayRowCol",
		xmRowColumnWidgetClass, changerRowCol,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	XtVaGetValues(abacus,
		XtNdisplayBase, &displayBase,
		XtNbase, &base, NULL);
#ifdef USE_SPIN
	createSpinner(changerDisplayRowCol, &displayBaseSlider,
		(char *) displayBaseLabel, displayBase, MIN_BASE, MAX_BASE, 2,
		(XtCallbackProc) displayBaseChangeCallback);
#else
	createSlider(changerDisplayRowCol, &displayBaseSlider,
		(char *) displayBaseLabel, displayBase, MIN_BASE, MAX_BASE, 2,
		(MAX_BASE - MIN_BASE + 1) * 4,
		(XtCallbackProc) displayBaseChangeCallback);
#endif
	changerAbacusRowCol = XtVaCreateManagedWidget("changerAbacusRowCol",
		xmRowColumnWidgetClass, changerRowCol,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(changerAbacusRowCol, &abacusBaseSlider,
		(char *) abacusBaseLabel, base, MIN_BASE, MAX_BASE, 2,
		(XtCallbackProc) abacusBaseChangeCallback);
#else
	createSlider(changerAbacusRowCol, &abacusBaseSlider,
		(char *) abacusBaseLabel, base, MIN_BASE, MAX_BASE, 2,
		(MAX_BASE - MIN_BASE + 1) * 4,
		(XtCallbackProc) abacusBaseChangeCallback);
#endif
	XtManageChild(pane);
	button = XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	return dialog;
}

static void
controlsMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);

#ifdef EXTRA
	if (value != 2)
#else
	if (value != 1)
#endif
		return;
	if (baseDialog != NULL)
		XtManageChild(baseDialog);
}

static void
beadControlMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);

	XtVaSetValues(abacus,
		XtNmenu, value + ACTION_CLEAR, NULL);
}

#ifdef EXTRA
static void
basicFormatMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);

	XtVaSetValues(abacus,
		XtNmenu, value + ACTION_FORMAT, NULL);
}
#endif

static void
displayFormatMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);

	XtVaSetValues(abacus,
		XtNmenu, value + ACTION_ROMAN_NUMERALS, NULL);
}

static void
specialRailsMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);

	XtVaSetValues(abacus,
		XtNmenu, value + ACTION_SIGN, NULL);
}

static void
secondaryRailsMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);

	XtVaSetValues(abacus,
		XtNmenu, value + ACTION_SUBDECK, NULL);
}

static void
teachOptionsMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);

	XtVaSetValues(abacus,
		XtNmenu, value + ACTION_RIGHT_TO_LEFT_ADD, NULL);
}

static Widget
createDelayChanger(Widget w, char *title, Boolean spin)
{
	Widget dialog, pane, button;
	Widget changerRowCol;
	int delay;
	char titleDsp[TITLE_FILE_LENGTH];
	XmString titleString = NULL;

	concatTitle(titleDsp, progDsp, ": ", title, TITLE_FILE_LENGTH);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(args[0], XmNdialogTitle, titleString);
	dialog = XmCreateMessageDialog(w, title, args, 1);
	XmStringFree(titleString);
	pane = XtVaCreateWidget("pane", xmPanedWindowWidgetClass, dialog,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	changerRowCol = XtVaCreateManagedWidget("changerRowCol",
		xmRowColumnWidgetClass, pane,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	XtVaGetValues(abacus,
		XtNdelay, &delay, NULL);
#ifdef USE_SPIN
	createSpinner(changerRowCol, &delaySlider,
		(char *) delayLabel, delay, 0, MAX_DELAY, 3,
		(XtCallbackProc) delayChangeCallback);
#else
	createSlider(changerRowCol, &delaySlider,
		(char *) delayLabel, delay, 0, MAX_DELAY, 3, SCROLL_SIZE,
		(XtCallbackProc) delayChangeCallback);
#endif
	XtManageChild(pane);
	button = XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	return dialog;
}

static void
effectsMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);

	if (value == 0) {
		XtVaSetValues(abacus,
			XtNmenu, value + ACTION_SOUND, NULL);
	} else if (value == 1) {
		if (delayDialog != NULL)
			XtManageChild(delayDialog);
	}
}

#elif defined(HAVE_ATHENA)

static void
clearText(Widget w, XtPointer clientData, XtPointer callData)
{
	Widget text = (Widget) clientData;

	XtVaSetValues(text,
		XtNstring, "", NULL);
	initializeText((AbacusWidget) abacus);
}

static void
calculateText(Widget w, XtPointer clientData, XtPointer callData)
{
	Widget text = (Widget) clientData;
	String str;

	XtVaGetValues(text,
		XtNstring, &str, NULL);
	XtVaSetValues(abacus,
		XtNaux, 0,
		XtNdeck, CALC_DECK,
		XtNmathBuffer, str, NULL);
#ifdef DEBUG
	(void) fprintf(stdout, "calcText: %s\n", str);
#endif
}

static void
teachText(Widget w, XtPointer clientData, XtPointer callData)
{
	Widget text = (Widget) clientData;
	String str;

	XtVaGetValues(text,
		XtNstring, &str, NULL);
	if (strlen(str) > 0)
		XtVaSetValues(abacus,
			XtNdeck, TEACH_DECK,
			XtNteachBuffer, str, NULL);
}

static void
abacusClearCancelCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	XtPopdown(clearDialog);
}

static void
createClearQuery(char *title, char *text)
{
	Widget dialog, okDialog, cancelDialog;

	clearDialog = XtCreatePopupShell(title,
		transientShellWidgetClass, topLevel, NULL, 0);
	dialog = XtVaCreateManagedWidget("dialog",
		dialogWidgetClass, clearDialog,
		XtNlabel, text, NULL);
	okDialog = XtVaCreateManagedWidget("OK",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(okDialog, XtNcallback,
		abacusClearCallback, dialog);
	cancelDialog = XtVaCreateManagedWidget("Cancel",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(cancelDialog, XtNcallback,
		abacusClearCancelCallback, dialog);
	XtRealizeWidget(clearDialog);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(clearDialog), &wmDeleteWindow, 1);
}
#endif

int
main(int argc, char **argv)
{
	int pixmapSize;
	Boolean demo;
	XtAppContext appCon;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	Widget controlPanel, menuBar;
	int mode, romanNumeralsMode, romanMarkersMode, piece, piecePercent;
	int bottomPiece, topPiece, bottomPiecePercent, topPiecePercent;
	int rails;
	Boolean latin, group, placeOnRail, vertical, stippleFrame;
	Boolean sign;
	unsigned int i;
#ifdef ANOMALY
	int anomalyMode;
#endif
	int subbase, anomaly, anomalySq;
	Boolean subdecksSeparated, altSubbeadPlacement;
	int museumMode;
	Boolean rightToLeftAdd, rightToLeftMult;
	int base = DEFAULT_BASE, displayBase = DEFAULT_BASE;
	Boolean sound, teach;
#ifdef LEE_ABACUS
	Boolean lee;
	Pixel leftAuxBeadColor, rightAuxBeadColor;
	int leftAuxRails, rightAuxRails;
#endif
#endif
#ifdef HAVE_MOTIF
	Boolean slot;
	int decimalPosition = DEFAULT_DECIMAL_POSITION;
	Widget pullDownMenu, popupMenu, widget;
	Widget menuBarPanel;
	Widget controlRowCol, trackerRowCol;
	Widget auxForm;
	Widget displayFormatMenu, specialRailsMenu;
	Widget teachMenu, effectsMenu;
	Widget learnMenu;
	XmString fileString, controlsString, learnString;
	XmString quitString, clearString, complementString;
	XmString undoString, redoString;
	XmString romanNumeralsString, romanMarkersString;
	XmString latinString, groupString, placeOnRailString, verticalString;
	XmString stippleFrameString, signString;
	XmString pieceStrings[PIECE_OPTIONS];
	XmString subdecksSeparatedString, altSubbeadPlacementString;
#ifdef ANOMALY
	XmString anomalyString;
	XmString anomalyStrings[ANOMALY_OPTIONS];
#endif
	XmString rightToLeftAddString, rightToLeftMultString;
	XmString soundString, delayString;
	XmString formatString, formatStrings[MAX_MODES], museumString;
	XmString museumStrings[MAX_MUSEUMS];
	XmString romanFormatStrings[ROMAN_MARKERS_FORMATS];
	XmString beadControlString;
#ifdef EXTRA
	XmString basicFormatString;
	XmString modeString, incrementString, decrementString;
#endif
	XmString baseSettingsString, displayFormatString;
	XmString specialRailsString, secondaryRailsString;
	XmString pieceRailString, piecePercentRailString;
	XmString subdeckRailString;
	XmString demoString, teachString;
	XmString teachOptionsString, effectsString;
#elif defined(HAVE_ATHENA)
	XtActionsRec actions[] = {
		{(char *) "DeleteWindowProc", deleteWindowProc},
		{(char *) "ClosePopupPanel", (XtActionProc) closePopupPanel}
	};
	String fallbackResources[] = {
		(char *) "?.translations: #override <Message>WM_PROTOCOLS: DeleteWindowProc()",
		(char *) "?.TransientShell.translations: #override <Message>WM_PROTOCOLS: ClosePopupPanel()",
		NULL
	};
	int delay;
	char titleDsp[TITLE_FILE_LENGTH];
	char *defaultString;
	Widget fileLabel, controlsLabel, learnLabel, helpLabel;
	Widget railBox, formatBox, romanNumeralsBox, romanMarkersBox;
	Widget pieceRailBox, piecePercentRailBox;
	Widget w;
#ifdef ANOMALY
	Widget anomalyBox;
#endif
	Widget subdeckRailBox, museumBox;
	Widget displayBaseBox, abacusBaseBox, delayBox;
	Widget groupBox, placeOnRailBox, verticalBox;
	Widget stippleFrameBox, signBox;
	Widget subdecksSeparatedBox, altSubbeadPlacementBox;
	Widget soundBox;
	Widget rightToLeftAddBox, rightToLeftMultBox;
	Widget railRowCol;
	Widget teachPaned, teachForm, teachROForm;
	Widget teachClearButton, teachButton, teachQuitButton;
	Widget calculatePaned, calculateForm, calculateTextField;
	Widget calculateClearButton, calculateButton, calculateQuitButton;
	Widget displayForm, displayQuitButton;
	Widget specialRailsForm, specialRailsQuitButton;
	Widget secondaryRailsForm, secondaryRailsQuitButton;
	Widget setupForm, setupQuitButton;
	int height;
#ifdef LEE_ABACUS
	Widget auxPanel, abacusAuxForm, trackerAuxForm;
#endif
#endif

#ifdef __VMS
	int n;
	progDsp = strrchr(argv[0], ']');
	for (n = 0; progDsp[n] != '\0' && progDsp[n] != '.'; n++);
	progDsp[n] = '\0';
#else
	progDsp = strrchr(argv[0], '/');
#endif
	if (progDsp != NULL)
		progDsp++;
	if (progDsp == NULL)
		progDsp = argv[0];
	topLevel = XtVaAppInitialize(NULL, "XAbacus",
		options, XtNumber(options), &argc, argv,
#ifdef HAVE_ATHENA
		fallbackResources,
#else
		NULL,
#endif
		NULL);
	appCon = XtWidgetToApplicationContext(topLevel);
	if (argc != 1)
		usage(argv[0], synopsisHelp);
#ifdef HAVE_MOTIF
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	controlsString = XmStringCreateSimple((char *) "Controls");
	learnString = XmStringCreateSimple((char *) "Learn");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, controlsString, 'C',
		XmVaCASCADEBUTTON, learnString, 'L', NULL);
	XmStringFree(fileString);
	XmStringFree(controlsString);
	quitString = XmStringCreateSimple((char *) "Exit");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "fileMenu",
		0, fileMenuCallback,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'x', NULL, NULL, NULL);
	XmStringFree(quitString);
	beadControlString = XmStringCreateSimple((char *) "Bead Control");
	baseSettingsString = XmStringCreateSimple((char *) "Base Settings...");
	displayFormatString = XmStringCreateSimple((char *) "Display Format");
	specialRailsString = XmStringCreateSimple((char *) "Special Rails");
	teachOptionsString = XmStringCreateSimple((char *) "Teach Options");
	effectsString = XmStringCreateSimple((char *) "Effects");
	clearString = XmStringCreateSimple((char *) "Clear");
	complementString = XmStringCreateSimple((char *) "Complement -");
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
#ifdef EXTRA
	basicFormatString = XmStringCreateSimple((char *) "Basic Format");
	modeString = XmStringCreateSimple((char *) "Format");
	incrementString = XmStringCreateSimple((char *) "Increment");
	int index = PIECE_BASE(value);
	decrementString = XmStringCreateSimple((char *) "Decrement");
#endif
	romanNumeralsString = XmStringCreateSimple((char *) "Roman Nvmerals");
	romanMarkersString = XmStringCreateSimple((char *) "Roman Markerz");
	for (i = 0; i < ROMAN_MARKERS_FORMATS; i++)
		romanFormatStrings[i] =
			XmStringCreateSimple((char *) romanFormatChoices[i]);
	latinString = XmStringCreateSimple((char *) "Latin ~");
	groupString = XmStringCreateSimple((char *) "Group");
	placeOnRailString = XmStringCreateSimple((char *) "PlaceOnRail #");
	verticalString = XmStringCreateSimple((char *) "Vertical |");
	stippleFrameString = XmStringCreateSimple((char *) "FrameStipple x");
	signString = XmStringCreateSimple((char *) "Sign");
	for (i = 0; i < PIECE_OPTIONS; i++)
		pieceStrings[i] =
			XmStringCreateSimple((char *) pieceChoices[i]);
	pieceRailString = XmStringCreateSimple((char *) "Piece Rail");
	piecePercentRailString = XmStringCreateSimple((char *) "Piece Percent Rail");
	secondaryRailsString = XmStringCreateSimple((char *) "Secondary Rails");
	subdeckRailString = XmStringCreateSimple((char *) "Subdeck Rail");
	museumString = XmStringCreateSimple((char *) "Museum");
	for (i = 0; i < MAX_MUSEUMS; i++)
		museumStrings[i] =
			XmStringCreateSimple((char *) museumChoices[i]);
	subdecksSeparatedString = XmStringCreateSimple((char *) "Subdecks Separated");
	altSubbeadPlacementString = XmStringCreateSimple((char *) "Alt Subbead Placement");
#ifdef ANOMALY
	anomalyString = XmStringCreateSimple((char *) "Anomaly");
	for (i = 0; i < ANOMALY_OPTIONS; i++)
		anomalyStrings[i] =
			XmStringCreateSimple((char *) anomalyChoices[i]);
#endif
	rightToLeftAddString = XmStringCreateSimple((char *) "Right To Left Add +");
	rightToLeftMultString = XmStringCreateSimple((char *) "Right To Left Mult *");
	soundString = XmStringCreateSimple((char *) "Sound @");
	delayString = XmStringCreateSimple((char *) "Delay...");
	demoString = XmStringCreateSimple((char *) "Demo");
	teachString = XmStringCreateSimple((char *) "Teach $");
	popupMenu = XmVaCreateSimplePulldownMenu(menuBar, (char *) "controlsMenu",
		1, controlsMenuCallback,
		XmVaCASCADEBUTTON, beadControlString, 'B',
#ifdef EXTRA
		XmVaCASCADEBUTTON, basicFormatString, 'F',
#endif
		XmVaPUSHBUTTON, baseSettingsString, 'S', NULL, NULL,
		XmVaCASCADEBUTTON, displayFormatString, 'D',
		XmVaCASCADEBUTTON, specialRailsString, 'R',
		XmVaCASCADEBUTTON, teachOptionsString, 'T',
		XmVaCASCADEBUTTON, effectsString, 'E', NULL);
	XmStringFree(beadControlString);
#ifdef EXTRA
	XmStringFree(basicFormatString);
#endif
	XmStringFree(baseSettingsString);
	XmStringFree(displayFormatString);
	XmStringFree(specialRailsString);
	XmStringFree(teachOptionsString);
	XmStringFree(effectsString);
	(void) XmVaCreateSimplePulldownMenu(popupMenu, (char *) "BeadControlMenu",
		0, beadControlMenuCallback,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, complementString, '-', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL, NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
	XmStringFree(complementString);
#ifdef EXTRA
	(void) XmVaCreateSimplePulldownMenu(popupMenu, (char *) "BasicFormatMenu",
		1, basicFormatMenuCallback,
		XmVaPUSHBUTTON, modeString, 'F', NULL, NULL,
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL, NULL);
	XmStringFree(modeString);
	XmStringFree(incrementString);
	XmStringFree(decrementString);
#endif
	displayFormatMenu = XmVaCreateSimplePulldownMenu(popupMenu, (char *) "DisplayFormatMenu",
#ifdef EXTRA
		1 +
#endif
		2, displayFormatMenuCallback,
		XmVaCASCADEBUTTON, romanNumeralsString, 'v',
		XmVaCASCADEBUTTON, romanMarkersString, 'z',
		XmVaTOGGLEBUTTON, latinString, '~', NULL, NULL,
		XmVaTOGGLEBUTTON, groupString, 'G', NULL, NULL,
		XmVaTOGGLEBUTTON, placeOnRailString, '#', NULL, NULL,
		XmVaTOGGLEBUTTON, verticalString, '|', NULL, NULL,
		XmVaTOGGLEBUTTON, stippleFrameString, 'x', NULL, NULL, NULL);
	XmStringFree(romanNumeralsString);
	XmStringFree(romanMarkersString);
	XmStringFree(latinString);
	XmStringFree(groupString);
	XmStringFree(placeOnRailString);
	XmStringFree(verticalString);
	XmStringFree(stippleFrameString);
	romanNumeralsMenu = XmVaCreateSimplePulldownMenu(displayFormatMenu, (char *) "RomanNumeralsMenu",
		0, romanNumeralsMenuCallback,
		XmVaRADIOBUTTON, romanFormatStrings[NONE], NULL, NULL, NULL,
		XmVaRADIOBUTTON, romanFormatStrings[ANCIENT], NULL, NULL, NULL,
		XmVaRADIOBUTTON, romanFormatStrings[MODERN], NULL, NULL, NULL,
		XmNradioBehavior, True,
		XmNradioAlwaysOne, True, NULL);
	romanMarkersMenu = XmVaCreateSimplePulldownMenu(displayFormatMenu, (char *) "RomanMarkersMenu",
		1, romanMarkersMenuCallback,
		XmVaRADIOBUTTON, romanFormatStrings[NONE], NULL, NULL, NULL,
		XmVaRADIOBUTTON, romanFormatStrings[ANCIENT], NULL, NULL, NULL,
		XmVaRADIOBUTTON, romanFormatStrings[MODERN], NULL, NULL, NULL,
		XmVaRADIOBUTTON, romanFormatStrings[LATE], NULL, NULL, NULL,
		XmVaRADIOBUTTON, romanFormatStrings[ALT], NULL, NULL, NULL,
		XmNradioBehavior, True,
		XmNradioAlwaysOne, True, NULL);
	for (i = 0; i < ROMAN_MARKERS_FORMATS; i++)
		XmStringFree(romanFormatStrings[i]);
	specialRailsMenu = XmVaCreateSimplePulldownMenu(popupMenu, (char *) "SpecialRailsMenu",
#ifdef EXTRA
		1 +
#endif
		3, specialRailsMenuCallback,
		XmVaTOGGLEBUTTON, signString, 'S', NULL, NULL,
		XmVaCASCADEBUTTON, pieceRailString, 'P',
		XmVaCASCADEBUTTON, piecePercentRailString, 't',
		XmVaCASCADEBUTTON, secondaryRailsString, 'R',
#ifdef ANOMALY
		XmVaCASCADEBUTTON, anomalyString, 'a',
#endif
		NULL);
	XmStringFree(signString);
	XmStringFree(pieceRailString);
	XmStringFree(piecePercentRailString);
	XmStringFree(secondaryRailsString);
#ifdef ANOMALY
	XmStringFree(anomalyString);
#endif
	pieceRailMenu = XmVaCreateSimplePulldownMenu(specialRailsMenu, (char *) "PieceRailMenu",
		1, pieceRailMenuCallback,
		XmVaRADIOBUTTON, pieceStrings[NONE], NULL, NULL, NULL,
		XmVaRADIOBUTTON, pieceStrings[QUARTER_INDEX], NULL, NULL, NULL,
		XmVaRADIOBUTTON, pieceStrings[EIGHTH_INDEX], NULL, NULL, NULL,
		XmVaRADIOBUTTON, pieceStrings[TWELFTH_INDEX], NULL, NULL, NULL,
		XmNradioBehavior, True,
		XmNradioAlwaysOne, True, NULL);
	piecePercentRailMenu = XmVaCreateSimplePulldownMenu(specialRailsMenu, (char *) "PiecePercentRailMenu",
		2, piecePercentRailMenuCallback,
		XmVaRADIOBUTTON, pieceStrings[NONE], NULL, NULL, NULL,
		XmVaRADIOBUTTON, pieceStrings[QUARTER_INDEX], NULL, NULL, NULL,
		XmVaRADIOBUTTON, pieceStrings[EIGHTH_INDEX], NULL, NULL, NULL,
		XmVaRADIOBUTTON, pieceStrings[TWELFTH_INDEX], NULL, NULL, NULL,
		XmNradioBehavior, True,
		XmNradioAlwaysOne, True, NULL);
	secondaryRailsMenu = XmVaCreateSimplePulldownMenu(specialRailsMenu, (char *) "SecondaryRailsMenu",
		3, secondaryRailsMenuCallback,
		XmVaCASCADEBUTTON, subdeckRailString, 'b',
		XmVaTOGGLEBUTTON, subdecksSeparatedString, 'k', NULL, NULL,
		XmVaTOGGLEBUTTON, altSubbeadPlacementString, 'l', NULL, NULL,
		XmVaCASCADEBUTTON, museumString, 'M', NULL);
	XmStringFree(subdeckRailString);
	XmStringFree(subdecksSeparatedString);
	XmStringFree(altSubbeadPlacementString);
	XmStringFree(museumString);
	subdeckRailMenu = XmVaCreateSimplePulldownMenu(secondaryRailsMenu, (char *) "SubdeckRailMenu",
		0, subdeckRailMenuCallback,
		XmVaRADIOBUTTON, pieceStrings[NONE], NULL, NULL, NULL,
		XmVaRADIOBUTTON, pieceStrings[QUARTER_INDEX], NULL, NULL, NULL,
		XmVaRADIOBUTTON, pieceStrings[EIGHTH_INDEX], NULL, NULL, NULL,
		XmVaRADIOBUTTON, pieceStrings[TWELFTH_INDEX], NULL, NULL, NULL,
		XmNradioBehavior, True,
		XmNradioAlwaysOne, True, NULL);
	for (i = 0; i < PIECE_OPTIONS; i++)
		XmStringFree(pieceStrings[i]);
	museumMenu = XmVaCreateSimplePulldownMenu(secondaryRailsMenu, (char *) "MuseumMenu",
		3, museumMenuCallback,
		XmVaRADIOBUTTON, museumStrings[IT], NULL, NULL, NULL,
		XmVaRADIOBUTTON, museumStrings[UK], NULL, NULL, NULL,
		XmVaRADIOBUTTON, museumStrings[FR], NULL, NULL, NULL,
		XmNradioBehavior, True,
		XmNradioAlwaysOne, True, NULL);
	for (i = 0; i < MAX_MUSEUMS; i++)
		XmStringFree(museumStrings[i]);
#ifdef ANOMALY
	anomalyMenu = XmVaCreateSimplePulldownMenu(specialRailsMenu, (char *) "AnomalyMenu",
		5, anomalyMenuCallback,
		XmVaRADIOBUTTON, anomalyStrings[NONE], NULL, NULL, NULL,
		XmVaRADIOBUTTON, anomalyStrings[CALENDAR_INDEX], NULL, NULL, NULL,
		XmVaRADIOBUTTON, anomalyStrings[WATCH_INDEX], NULL, NULL, NULL,
		XmNradioBehavior, True,
		XmNradioAlwaysOne, True, NULL);
	for (i = 0; i < ANOMALY_OPTIONS; i++)
		XmStringFree(anomalyStrings[i]);
#endif
	teachMenu = XmVaCreateSimplePulldownMenu(popupMenu, (char *) "TeachMenu",
#ifdef EXTRA
		1 +
#endif
		4, teachOptionsMenuCallback,
		XmVaTOGGLEBUTTON, rightToLeftAddString, '+', NULL, NULL,
		XmVaTOGGLEBUTTON, rightToLeftMultString, '*', NULL, NULL, NULL);
	XmStringFree(rightToLeftAddString);
	XmStringFree(rightToLeftMultString);
	effectsMenu = XmVaCreateSimplePulldownMenu(popupMenu, (char *) "EffectsMenu",
#ifdef EXTRA
		1 +
#endif
		5, effectsMenuCallback,
		XmVaTOGGLEBUTTON, soundString, '@', NULL, NULL,
		XmVaPUSHBUTTON, delayString, 'D', NULL, NULL, NULL);
	XmStringFree(soundString);
	XmStringFree(delayString);
	learnString = XmStringCreateSimple((char *) "Learn");
	learnMenu = XmVaCreateSimplePulldownMenu(menuBar, (char *) "LearnMenu",
		2, learnMenuCallback,
		XmVaTOGGLEBUTTON, demoString, 'o', NULL, NULL,
		XmVaTOGGLEBUTTON, teachString, '$', NULL, NULL, NULL);
	XmStringFree(demoString);
	XmStringFree(teachString);
	XmStringFree(learnString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL); /* mnemonic warning on Cygwin */
	XtVaSetValues(menuBar,
		XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 0);
	widget = XtVaCreateManagedWidget("Features...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 1);
	widget = XtVaCreateManagedWidget("Synopsis...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'S', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 2);
	widget = XtVaCreateManagedWidget("Options...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 3);
	widget = XtVaCreateManagedWidget("References...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 4);
	widget = XtVaCreateManagedWidget("About...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 5);
	XtManageChild(menuBar);
	clearDialog = createQuery(topLevel, (char *) "Clear Query",
		(char *) "Are you sure you want to destroy the calculation?",
		(XtCallbackProc) abacusClearCallback);
	pauseDialog = createPause(topLevel, (char *) "Pause Calculation",
		(char *) "Ready to continue?",
		(XtCallbackProc) abacusPauseCallback);
	mainPanel = XtVaCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel, NULL);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate and out of date. */
		Widget bitmapRowCol;
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		bitmapRowCol = XtVaCreateManagedWidget("bitmapRowCol",
			xmRowColumnWidgetClass, controlPanel,
			XmNnumColumns, 4,
			XmNpacking, XmPACK_TIGHT, NULL);
		(void) XtVaGetValues(bitmapRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(bitmapRowCol),
			RootWindowOfScreen(XtScreen(bitmapRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(bitmapRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(bitmapRowCol),
			RootWindowOfScreen(XtScreen(bitmapRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(bitmapRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, bitmapRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move bead", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, bitmapRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, bitmapRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "    Clear", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, bitmapRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	controlRowCol = XtVaCreateManagedWidget("controlRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT,
		XmNisAligned, True, NULL);
	auxForm = XtVaCreateManagedWidget("auxForm",
		xmFormWidgetClass, mainPanel, NULL);
	abacus = XtVaCreateManagedWidget("abacus",
#ifdef LEE_KO
		XtNformat, "Korean",
#endif
		abacusWidgetClass, auxForm,
		XmNbottomAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_POSITION, NULL);
	trackerRowCol = XtVaCreateManagedWidget("trackerRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
#ifdef LEE_ABACUS
	XtVaGetValues(abacus,
		XtNlee, &lee,
		XtNleftAuxBeadColor, &leftAuxBeadColor,
		XtNrightAuxBeadColor, &rightAuxBeadColor,
		XtNleftAuxRails, &leftAuxRails,
		XtNrightAuxRails, &rightAuxRails,
		XtNdecimalPosition, &decimalPosition,
		XtNplaceOnRail, &placeOnRail,
		XtNbase, &base,
		XtNdisplayBase, &displayBase, NULL);
	if (lee) {
		Widget auxTrackerForm;

		XtVaSetValues(auxForm,
			XmNfractionBase, leftAuxRails + rightAuxRails, NULL);
		auxTrackerForm = XtVaCreateManagedWidget("auxTrackerRowCol",
			xmFormWidgetClass, trackerRowCol,
			XmNfractionBase, leftAuxRails + rightAuxRails, NULL);
		leftAuxTracker = XtVaCreateManagedWidget("0.0",
			xmTextWidgetClass, auxTrackerForm,
			XmNtopAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_POSITION,
			XmNbottomAttachment, XmATTACH_FORM,
			XmNrightPosition, leftAuxRails, NULL);
		XtAddCallback(leftAuxTracker, XmNactivateCallback,
			(XtCallbackProc) abacusMathCallback, (XtPointer) NULL);
		rightAuxTracker = XtVaCreateManagedWidget("0.0",
			xmTextWidgetClass, auxTrackerForm,
			XmNtopAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_WIDGET,
			XmNbottomAttachment, XmATTACH_FORM,
			XmNleftWidget, leftAuxTracker, NULL);
		XtAddCallback(rightAuxTracker, XmNactivateCallback,
			(XtCallbackProc) abacusMathCallback, (XtPointer) NULL);
	}
#endif
	tracker = XtVaCreateManagedWidget("0.0",
		xmTextWidgetClass, trackerRowCol, NULL);
	XtAddCallback(tracker, XmNactivateCallback,
		(XtCallbackProc) abacusMathCallback, (XtPointer) NULL);
#ifdef LEE_ABACUS
	if (lee) {
		XtVaSetValues(abacus,
			XmNtopPosition, 5, NULL);
	}
	if (lee) {
		leftAuxAbacus = XtVaCreateManagedWidget("leftAuxAbacus",
			abacusWidgetClass, auxForm,
			XmNtopAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_FORM,
			XmNbottomAttachment, XmATTACH_WIDGET,
			XmNrightAttachment, XmATTACH_POSITION,
			XmNbottomWidget, abacus,
			XtNformat, "Japanese",
			XtNaux, LEFT_AUX,
			XtNprimaryBeadColor, leftAuxBeadColor,
			XmNrightPosition, leftAuxRails,
			XtNrails, leftAuxRails,
			XtNdecimalPosition, decimalPosition,
			XtNplaceOnRail, placeOnRail,
			XtNbase, base,
			XtNdisplayBase, displayBase, NULL);
		XtAddCallback(leftAuxAbacus, XtNselectCallback,
			(XtCallbackProc) abacusCallback, (XtPointer) NULL);
		rightAuxAbacus = XtVaCreateManagedWidget("rightAuxAbacus",
			abacusWidgetClass, auxForm,
			XmNtopAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			XmNbottomAttachment, XmATTACH_WIDGET,
			XmNbottomWidget, abacus,
			XmNleftAttachment, XmATTACH_WIDGET,
			XmNleftWidget, leftAuxAbacus,
			XtNformat, "Japanese",
			XtNaux, RIGHT_AUX,
			XtNprimaryBeadColor, rightAuxBeadColor,
			XtNrails, rightAuxRails,
			XtNdecimalPosition, decimalPosition,
			XtNplaceOnRail, placeOnRail,
			XtNbase, base,
			XtNdisplayBase, displayBase, NULL);
		XtAddCallback(rightAuxAbacus, XtNselectCallback,
			(XtCallbackProc) abacusCallback, (XtPointer) NULL);
	}
#endif
#elif defined(HAVE_ATHENA)
	XtAppAddActions(appCon, actions, XtNumber(actions));
	createClearQuery((char *) "Clear Query",
		(char *) "Are you sure you want to destroy the calculation?");
	strncpy(descriptionHelp, description1Help, DESCRIPTION1_SIZE);
	strncat(descriptionHelp, "\n", 2);
	strncat(descriptionHelp, description2Help, DESCRIPTION2_SIZE);
	createHelp(topLevel, &descriptionDialog, (char *) "Description",
		(char *) descriptionHelp, (XtCallbackProc) closePopupPanel2);
	createScrollHelp(topLevel, &featuresDialog, (char *) "Features",
		(char *) featuresHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &synopsisDialog, (char *) "Synopsis",
		(char *) synopsisHelp, (XtCallbackProc) closePopupPanel2);
	strncpy(optionsHelp, options1Help, OPTIONS1_SIZE);
	strncat(optionsHelp, "\n", 2);
	strncat(optionsHelp, options2Help, OPTIONS2_SIZE);
	strncat(optionsHelp, "\n", 2);
	strncat(optionsHelp, options3Help, OPTIONS3_SIZE);
	strncat(optionsHelp, "\n", 2);
	createScrollHelp(topLevel, &optionsDialog, (char *) "Options",
		(char *) optionsHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &referencesDialog, (char *) "References",
		(char *) referencesHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &aboutDialog, (char *) "About",
		(char *) aboutHelp, (XtCallbackProc) closePopupPanel2);

	calculateDialogShell = XtCreatePopupShell("Calculate",
		transientShellWidgetClass, topLevel, NULL, 0);
	calculatePaned = XtVaCreateManagedWidget("dialog",
		panedWidgetClass, calculateDialogShell, NULL);
	calculateForm = XtVaCreateManagedWidget("form",
		formWidgetClass, calculatePaned, NULL);
	calculateClearButton = XtVaCreateManagedWidget("clear",
		commandWidgetClass, calculateForm,
		XtNlabel, "Clear", NULL);
	calculateButton = XtVaCreateManagedWidget("calculate",
		commandWidgetClass, calculateForm,
		XtNlabel, "Calculate",
		XtNfromHoriz, calculateClearButton, NULL);
	calculateQuitButton = XtVaCreateManagedWidget("Quit",
		commandWidgetClass, calculateForm,
		XtNfromHoriz, calculateButton, NULL);
	calculateTextField = XtVaCreateManagedWidget("text",
		asciiTextWidgetClass, calculatePaned,
		XtNstring, "0.0",
		XtNshowGrip, False,
		XtNeditType, XawtextEdit,
		XtNjustify, XtJustifyRight,
		XtNscrollVertical, XawtextScrollAlways,
		XtNfromVert, calculateForm, NULL);
	XtAddCallback(calculateClearButton,
		XtNcallback, clearText, (XtPointer) calculateTextField);
	XtAddCallback(calculateButton,
		XtNcallback, calculateText, (XtPointer) calculateTextField);
	XtAddCallback(calculateQuitButton,
		XtNcallback, (XtCallbackProc) closePopupPanel3,
		(XtPointer) calculateTextField);

	teachDialogShell = XtCreatePopupShell("Teach",
		transientShellWidgetClass, topLevel, NULL, 0);
	teachPaned = XtVaCreateManagedWidget("dialog",
		panedWidgetClass, teachDialogShell, NULL);
	teachForm = XtVaCreateManagedWidget("form",
		formWidgetClass, teachPaned, NULL);
	teachClearButton = XtVaCreateManagedWidget("clear",
		commandWidgetClass, teachForm,
		XtNlabel, "Clear", NULL);
	teachButton = XtVaCreateManagedWidget("teach",
		commandWidgetClass, teachForm,
		XtNlabel, "Teach",
		XtNfromHoriz, teachClearButton, NULL);
	teachQuitButton = XtVaCreateManagedWidget("Quit",
		commandWidgetClass, teachForm,
		XtNfromHoriz, teachButton, NULL);
	teachTracker = XtVaCreateManagedWidget("text",
		asciiTextWidgetClass, teachPaned,
		XtNstring, "0.0",
		XtNshowGrip, True,
		XtNeditType, XawtextEdit,
		XtNjustify, XtJustifyRight,
		XtNscrollVertical, XawtextScrollAlways,
		XtNfromVert, teachForm, NULL);
	teachROForm = XtVaCreateManagedWidget("roform",
		formWidgetClass, teachPaned,
		XtNfromVert, teachTracker, NULL);
	teachLine[0] = XtVaCreateManagedWidget("teachText1",
		labelWidgetClass, teachROForm,
		XtNlabel, "		      ",
		XtNborderWidth, 0,
		XtNjustify, XtJustifyLeft, NULL);
	teachLine[1] = XtVaCreateManagedWidget("teachText2",
		labelWidgetClass, teachROForm,
		XtNlabel, "		      ",
		XtNborderWidth, 0,
		XtNjustify, XtJustifyLeft,
		XtNfromVert, teachLine[0], NULL);
	teachLine[2] = XtVaCreateManagedWidget("teachText3",
		labelWidgetClass, teachROForm,
		XtNlabel, "		      ",
		XtNborderWidth, 0,
		XtNjustify, XtJustifyLeft,
		XtNfromVert, teachLine[1], NULL);
	XtAddCallback(teachClearButton,
		XtNcallback, clearText, (XtPointer) teachTracker);
	XtAddCallback(teachButton,
		XtNcallback, teachText, (XtPointer) teachTracker);
	XtAddCallback(teachQuitButton,
		XtNcallback, (XtCallbackProc) closePopupPanel3,
		(XtPointer) teachTracker);

	concatTitle(titleDsp, progDsp, ": ", "Setup", TITLE_FILE_LENGTH);
	displayDialogShell = XtCreatePopupShell(titleDsp,
		transientShellWidgetClass, topLevel, NULL, 0);
	displayForm = XtVaCreateManagedWidget("form",
		formWidgetClass, displayDialogShell, NULL);
	romanNumeralsBox = XtVaCreateManagedWidget("romanNumeralsMenu",
		boxWidgetClass, displayForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	romanMarkersBox = XtVaCreateManagedWidget("romanMarkersMenu",
		boxWidgetClass, displayForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, romanNumeralsBox, NULL);
	latinBox = XtVaCreateManagedWidget("latin",
		boxWidgetClass, displayForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, romanMarkersBox, NULL);
	groupBox = XtVaCreateManagedWidget("group",
		boxWidgetClass, displayForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, latinBox, NULL);
	placeOnRailBox = XtVaCreateManagedWidget("placeOnRail",
		boxWidgetClass, displayForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, groupBox, NULL);
	verticalBox = XtVaCreateManagedWidget("vertical",
		boxWidgetClass, displayForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, placeOnRailBox, NULL);
	stippleFrameBox = XtVaCreateManagedWidget("stippleFrame",
		boxWidgetClass, displayForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, verticalBox, NULL);
	displayQuitButton = XtVaCreateManagedWidget("Quit",
		commandWidgetClass, displayForm,
		XtNfromVert, stippleFrameBox, NULL);
	XtAddCallback(displayQuitButton,
		XtNcallback, (XtCallbackProc) closePopupPanel2, NULL);
	specialRailsDialogShell = XtCreatePopupShell(titleDsp,
		transientShellWidgetClass, topLevel, NULL, 0);
	specialRailsForm = XtVaCreateManagedWidget("form",
		formWidgetClass, specialRailsDialogShell, NULL);
	signBox = XtVaCreateManagedWidget("Sign",
		boxWidgetClass, specialRailsForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	pieceRailBox = XtVaCreateManagedWidget("pieceRailBox",
		boxWidgetClass, specialRailsForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, signBox, NULL);
	piecePercentRailBox = XtVaCreateManagedWidget("piecePercentRailBox",
		boxWidgetClass, specialRailsForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, pieceRailBox, NULL);
#ifdef ANOMALY
	anomalyBox = XtVaCreateManagedWidget("anomalyBox",
		boxWidgetClass, specialRailsForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, piecePercentRailBox, NULL);
	specialRailsQuitButton = XtVaCreateManagedWidget("Quit",
		commandWidgetClass, specialRailsForm,
		XtNfromVert, anomalyBox, NULL);
#else
	specialRailsQuitButton = XtVaCreateManagedWidget("Quit",
		commandWidgetClass, specialRailsForm,
		XtNfromVert, piecePercentRailBox, NULL);
#endif
	XtAddCallback(specialRailsQuitButton,
		XtNcallback, (XtCallbackProc) closePopupPanel2, NULL);
	secondaryRailsDialogShell = XtCreatePopupShell(titleDsp,
		transientShellWidgetClass, topLevel, NULL, 0);
	secondaryRailsForm = XtVaCreateManagedWidget("form",
		formWidgetClass, secondaryRailsDialogShell, NULL);
	subdeckRailBox = XtVaCreateManagedWidget("subdeckRailBox",
		boxWidgetClass, secondaryRailsForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	subdecksSeparatedBox = XtVaCreateManagedWidget("Subdecks Separated",
		boxWidgetClass, secondaryRailsForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, subdeckRailBox, NULL);
	altSubbeadPlacementBox = XtVaCreateManagedWidget("Alt Subbead Placement",
		boxWidgetClass, secondaryRailsForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, subdecksSeparatedBox, NULL);
	museumBox = XtVaCreateManagedWidget("museumBox",
		boxWidgetClass, secondaryRailsForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, altSubbeadPlacementBox, NULL);
	secondaryRailsQuitButton = XtVaCreateManagedWidget("Quit",
		commandWidgetClass, secondaryRailsForm,
		XtNfromVert, museumBox, NULL);
	XtAddCallback(secondaryRailsQuitButton,
		XtNcallback, (XtCallbackProc) closePopupPanel2, NULL);
	setupDialogShell = XtCreatePopupShell(titleDsp,
		transientShellWidgetClass, topLevel, NULL, 0);
	setupForm = XtVaCreateManagedWidget("form",
		formWidgetClass, setupDialogShell, NULL);
	displayBaseBox = XtVaCreateManagedWidget("displayBaseBox",
		boxWidgetClass, setupForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	abacusBaseBox = XtVaCreateManagedWidget("abacusBaseBox",
		boxWidgetClass, setupForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, displayBaseBox, NULL);
	delayBox = XtVaCreateManagedWidget("delayBox",
		boxWidgetClass, setupForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, abacusBaseBox, NULL);
	soundBox = XtVaCreateManagedWidget("soundBox",
		boxWidgetClass, setupForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, delayBox, NULL);
	rightToLeftAddBox = XtVaCreateManagedWidget("rightToLeftAddBox",
		boxWidgetClass, setupForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, soundBox, NULL);
	rightToLeftMultBox = XtVaCreateManagedWidget("rightToLeftMultBox",
		boxWidgetClass, setupForm,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromVert, rightToLeftAddBox, NULL);
	setupQuitButton = XtVaCreateManagedWidget("Quit",
		commandWidgetClass, setupForm,
		XtNfromVert, rightToLeftMultBox, NULL);
	XtAddCallback(setupQuitButton,
		XtNcallback, (XtCallbackProc) closePopupPanel2, NULL);
	mainPanel = XtVaCreateManagedWidget("mainPanel",
		panedWidgetClass, topLevel, NULL);
	menuBar = XtVaCreateManagedWidget("MenuBar",
		formWidgetClass, mainPanel,
		XtNborderWidth, 1, NULL);
	createMenu(menuBar, &fileLabel, NULL,
		fileTypes, "File", numFileTypes,
		0, False, fileMenuCallback);
	createMenu(menuBar, &controlsLabel, fileLabel,
		controlTypes, "Controls", numControlTypes,
		8, False, controlsMenuCallback);
	learnLabel = XtVaCreateManagedWidget("Learn",
		menuButtonWidgetClass, menuBar,
		XtNborderWidth, 0,
		XtNfromHoriz, controlsLabel, NULL);
	learnMenu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, learnLabel, NULL);
	for (i = 0; i < numLearnTypes; i++) {
		if (i == 1) {
			w = XtVaCreateManagedWidget(learnTypes[i],
				smeBSBObjectClass, learnMenu,
				XtNleftMargin, 20, NULL); /* for check */
			demoMenuItem = w;
		} else
			w = XtVaCreateManagedWidget(learnTypes[i],
				smeBSBObjectClass, learnMenu, NULL);
		XtAddCallback(w,
			XtNcallback, (XtCallbackProc) learnMenuCallback,
			(XtPointer) (size_t) i);
	}
	createMenu(menuBar, &helpLabel, learnLabel,
		helpTypes, "Help", numHelpTypes,
		0, False, helpMenuCallback);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		formWidgetClass, mainPanel,
		XtNborderWidth, 0, NULL);
	railRowCol = XtVaCreateManagedWidget("railRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0, NULL);
	railBox = XtVaCreateManagedWidget(NULL,
		boxWidgetClass, railRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	formatBox = XtVaCreateManagedWidget("formatMenu",
		formWidgetClass, railRowCol,
		XtNborderWidth, 0,
		XtNfromHoriz, railBox, NULL);
	createBlank(&defaultString, MAX_RAILS + 20 + 13 + 2, (char *) "0.0", 1);
	tracker = XtVaCreateManagedWidget(defaultString,
		labelWidgetClass, controlPanel,
#if 0
		/*dialogWidgetClass, controlPanel,*/
		asciiTextWidgetClass, controlPanel,
		XtNstring, defaultString,
		/*XtNlabel, defaultString,*/
		XtNeditType, XawtextEdit,
		XtNjustifyMode, XawjustifyRight,
		XtNjustify, XawjustifyRight,
#endif
		XtNjustify, XtJustifyRight,
		XtNfromVert, railRowCol, NULL);
	/*XtAddCallback(tracker, XtNcallback,
		(XtCallbackProc) abacusMathCallback, (XtPointer) NULL);*/
	free(defaultString);
	abacus = XtVaCreateManagedWidget("abacus",
		abacusWidgetClass, mainPanel,
#ifdef LEE_KO
		XtNformat, "Korean",
#endif
		XtNfromVert, tracker, NULL);
	XtVaGetValues(abacus,
		XtNlee, &lee,
		XtNheight, &height,
		XtNleftAuxBeadColor, &leftAuxBeadColor,
		XtNrightAuxBeadColor, &rightAuxBeadColor,
		XtNleftAuxRails, &leftAuxRails,
		XtNrightAuxRails, &rightAuxRails, NULL);
#ifdef LEE_ABACUS
	if (lee) {
		XtDestroyWidget(abacus); /* wrong order */
		XtDestroyWidget(tracker); /* wrong order */
		createBlank(&defaultString, MAX_RAILS + 13 + 2, (char *) "0.0", 1);
		trackerAuxForm = XtVaCreateManagedWidget("trackerAuxForm",
			formWidgetClass, controlPanel,
			XtNdefaultDistance, 0,
			XtNborderWidth, 0,
			XtNfromVert, railRowCol, NULL);
		leftAuxTracker = XtVaCreateManagedWidget(defaultString,
			labelWidgetClass, trackerAuxForm,
			XtNjustify, XtJustifyRight, NULL);
		rightAuxTracker = XtVaCreateManagedWidget(defaultString,
			labelWidgetClass, trackerAuxForm,
			XtNjustify, XtJustifyRight,
			XtNfromHoriz, leftAuxTracker, NULL);
		free(defaultString);
		createBlank(&defaultString, 2 * MAX_RAILS + 26 + 4, (char *) "0.0", 1);
		tracker = XtVaCreateManagedWidget(defaultString,
			labelWidgetClass, controlPanel,
			XtNjustify, XtJustifyRight,
			XtNfromVert, trackerAuxForm, NULL);
		free(defaultString);
		auxPanel = XtVaCreateManagedWidget("auxPanel",
			panedWidgetClass, mainPanel,
			XtNfromVert, controlPanel, NULL);
		abacusAuxForm = XtVaCreateManagedWidget("abacusAuxForm",
			formWidgetClass, auxPanel,
			XtNdefaultDistance, 0,
			XtNborderWidth, 0, NULL);
		leftAuxAbacus = XtVaCreateManagedWidget("leftAuxAbacus",
			abacusWidgetClass, abacusAuxForm,
			XtNformat, "Japanese",
			XtNaux, LEFT_AUX,
			XtNprimaryBeadColor, leftAuxBeadColor,
			XtNrails, leftAuxRails, NULL);
		XtAddCallback(leftAuxAbacus, XtNselectCallback,
			(XtCallbackProc) abacusCallback, (XtPointer) NULL);
		rightAuxAbacus = XtVaCreateManagedWidget("rightAuxAbacus",
			abacusWidgetClass, abacusAuxForm,
			XtNformat, "Japanese",
			XtNaux, RIGHT_AUX,
			XtNprimaryBeadColor, rightAuxBeadColor,
			XtNrails, rightAuxRails,
			XtNfromHoriz, leftAuxAbacus, NULL);
		XtAddCallback(rightAuxAbacus, XtNselectCallback,
			(XtCallbackProc) abacusCallback, (XtPointer) NULL);
		abacus = XtVaCreateManagedWidget("abacus",
			abacusWidgetClass, mainPanel,
			XtNheight, 2 * height,
#ifdef LEE_KO
			XtNformat, "Korean",
#endif
			XtNfromVert, abacusAuxForm, NULL);
	}
#endif
	XtVaGetValues(abacus,
		XtNmode, &mode,
		XtNrails, &rails,
		XtNromanNumeralsMode, &romanNumeralsMode,
		XtNromanMarkersMode, &romanMarkersMode,
		XtNlatin, &latin,
		XtNgroup, &group,
		XtNplaceOnRail, &placeOnRail,
		XtNvertical, &vertical,
		XtNstippleFrame, &stippleFrame,
		XtNsign, &sign,
		XtNtopPiece, &topPiece,
		XtNbottomPiece, &bottomPiece,
		XtNtopPiecePercent, &topPiecePercent,
		XtNbottomPiecePercent, &bottomPiecePercent,
		XtNsubdecksSeparated, &subdecksSeparated,
		XtNaltSubbeadPlacement, &altSubbeadPlacement,
		XtNmuseumMode, &museumMode,
		XtNanomaly, &anomaly,
		XtNanomalySq, &anomalySq,
		XtNrightToLeftAdd, &rightToLeftAdd,
		XtNrightToLeftMult, &rightToLeftMult,
		XtNsubbase, &subbase,
		XtNbase, &base,
		XtNdisplayBase, &displayBase,
		XtNdelay, &delay,
		XtNsound, &sound,
		XtNteach, &teach, NULL);
	piece = (topPiece == 0) ? bottomPiece : (topPiece * bottomPiece);
	piecePercent = (topPiecePercent == 0) ? bottomPiecePercent :
		(topPiecePercent * bottomPiecePercent);
#ifdef USE_SPIN
	createSpinner(railBox, &railSliderLabel,
		railLabel, 0, rails, MIN_RAILS, MAX_RAILS, False,
		railUpCallback, railDownCallback);
#else
	createSlider(railBox, &railSliderLabel, &railSlider,
		railLabel, 0, rails, MIN_RAILS, MAX_RAILS, False,
		(MAX_RAILS - MIN_RAILS + 1) * 6,
		railChangeCallback, railChangeCallback);
#endif
	createPopupMenu(romanNumeralsBox, &romanNumeralsLabel,
		romanFormatChoices, "Roman Nvmerals:", 16,
		romanNumeralsMode, ROMAN_NUMERALS_FORMATS,
		romanNumeralsMenuCallback);
	createPopupMenu(romanMarkersBox, &romanMarkersLabel,
		romanFormatChoices, "Roman Markerz:", 16,
		romanMarkersMode, ROMAN_MARKERS_FORMATS,
		romanMarkersMenuCallback);
	createToggle(latinBox, &latinToggle,
		"Latin ~:", 17,
		latin, (XtCallbackProc) latinCallback);
	createToggle(groupBox, &groupToggle,
		"Group:", 17,
		group, (XtCallbackProc) groupCallback);
	createToggle(placeOnRailBox, &placeOnRailToggle,
		"Place On Rail #:", 17,
		placeOnRail, (XtCallbackProc) placeOnRailCallback);
	createToggle(verticalBox, &verticalToggle,
		"Vertical |:", 17,
		vertical, (XtCallbackProc) verticalCallback);
	createToggle(stippleFrameBox, &stippleFrameToggle,
		"Stipple Frame x:", 17,
		stippleFrame, (XtCallbackProc) stippleFrameCallback);
	createToggle(signBox, &signToggle,
		"Sign:", 15,
		sign, (XtCallbackProc) signCallback);
	createPopupMenu(pieceRailBox, &pieceRailLabel,
		pieceChoices, "Piece:", 15,
		piece >> 2, PIECE_OPTIONS, pieceRailMenuCallback);
	createPopupMenu(piecePercentRailBox, &piecePercentRailLabel,
		pieceChoices, "Piece Percent:", 15,
		piecePercent >> 2, PIECE_OPTIONS,
		piecePercentRailMenuCallback);
#ifdef ANOMALY
	anomalyMode = ANOMALY_CHOICE(anomaly, anomalySq);
	createPopupMenu(anomalyBox, &anomalyLabel,
		anomalyChoices, "Anomaly:", 15,
		anomalyMode, ANOMALY_OPTIONS, anomalyMenuCallback);
#endif
	createPopupMenu(subdeckRailBox, &subdeckRailLabel,
		pieceChoices, "Subdeck:", 9,
		subbase >> 2, PIECE_OPTIONS, subdeckRailMenuCallback);
	createToggle(subdecksSeparatedBox, &subdecksSeparatedToggle,
		"Subdecks Separated:", 23,
		subdecksSeparated, (XtCallbackProc) subdecksSeparatedCallback);
	createToggle(altSubbeadPlacementBox, &altSubbeadPlacementToggle,
		"Alt Subbead Placement:", 23,
		altSubbeadPlacement, (XtCallbackProc) altSubbeadPlacementCallback);
	createPopupMenu(museumBox, &museumLabel,
		museumChoices, "Museum:", 9,
		museumMode, MAX_MUSEUMS, museumMenuCallback);
#ifdef USE_BASE_SPIN
	createSpinner(displayBaseBox, &displayBaseSliderLabel,
		displayBaseLabel, 14, displayBase,
		MIN_BASE, MAX_BASE, True,
		displayBaseUpCallback, displayBaseDownCallback);
	createSpinner(abacusBaseBox, &abacusBaseSliderLabel,
		abacusBaseLabel, 14, base,
		MIN_BASE, MAX_BASE, True,
		abacusBaseUpCallback, abacusBaseDownCallback);
#else
	createSlider(displayBaseBox, &displayBaseSliderLabel,
		&displayBaseSlider, displayBaseLabel, 14, displayBase,
		MIN_BASE, MAX_BASE, True, (MAX_BASE - MIN_BASE + 1) * 4,
		displayBaseChangeCallback, displayBaseChangeCallback);
	createSlider(abacusBaseBox, &abacusBaseSliderLabel,
		&abacusBaseSlider, abacusBaseLabel, 14, base,
		MIN_BASE, MAX_BASE, True, (MAX_BASE - MIN_BASE + 1) * 4,
		abacusBaseChangeCallback, abacusBaseChangeCallback);
#endif
#ifdef USE_SPIN
	createSpinner(delayBox, &delaySliderLabel,
		delayLabel, 14, delay, 0, MAX_DELAY, False,
		delayUpCallback, delayDownCallback);
#else
	createSlider(delayBox, &delaySliderLabel, &delaySlider,
		delayLabel, 14, delay, 0, MAX_DELAY, False, SCROLL_SIZE,
		delayChangeCallback, delayChangeCallback);
#endif
	createToggle(soundBox, &soundToggle,
		"Sound:", 26,
		sound, (XtCallbackProc) soundCallback);
	createToggle(rightToLeftAddBox, &rightToLeftAddToggle,
		"Teach Right To Left  Add:", 26,
		rightToLeftAdd, (XtCallbackProc) rightToLeftAddCallback);
	createToggle(rightToLeftMultBox, &rightToLeftMultToggle,
		"Teach Right To Left Mult:", 26,
		rightToLeftMult, (XtCallbackProc) rightToLeftMultCallback);
	createPopupMenu(formatBox, &formatLabel,
		formatChoices, "Format:", 0,
		mode, MAX_MODES, formatCallback);
	XtRealizeWidget(topLevel);
	setToggle(latinToggle, latin, False);
	setToggle(groupToggle, group, False);
	setToggle(placeOnRailToggle, placeOnRail, False);
	setToggle(verticalToggle, vertical, False);
	setToggle(stippleFrameToggle, stippleFrame, False);
	setToggle(signToggle, sign, False);
	setToggle(subdecksSeparatedToggle, subdecksSeparated, False);
	setToggle(altSubbeadPlacementToggle, altSubbeadPlacement, False);
	setToggle(rightToLeftAddToggle, rightToLeftAdd, False);
	setToggle(rightToLeftMultToggle, rightToLeftMult, False);
	setToggle(soundToggle, sound, False);
	setDemoCheck(demoMenuItem);
	checkEnabled(abacus);
	if (wmDeleteWindow == None)
		wmDeleteWindow = XInternAtom(XtDisplay(topLevel),
			"WM_DELETE_WINDOW", FALSE);
	XSetWMProtocols(XtDisplay(topLevel), XtWindow(topLevel),
		&wmDeleteWindow, 1);
#else
	abacus = XtVaCreateManagedWidget("abacus",
		abacusWidgetClass, topLevel, NULL);
#endif
	XtVaGetValues(abacus,
#ifdef HAVE_MOTIF
		XtNmode, &mode,
		XtNrails, &rails,
		XtNromanNumeralsMode, &romanNumeralsMode,
		XtNromanMarkersMode, &romanMarkersMode,
		XtNlatin, &latin,
		XtNgroup, &group,
		XtNplaceOnRail, &placeOnRail,
		XtNvertical, &vertical,
		XtNstippleFrame, &stippleFrame,
		XtNsign, &sign,
		XtNbottomPiece, &bottomPiece,
		XtNtopPiece, &topPiece,
		XtNbottomPiecePercent, &bottomPiecePercent,
		XtNtopPiecePercent, &topPiecePercent,
		XtNsubdecksSeparated, &subdecksSeparated,
		XtNaltSubbeadPlacement, &altSubbeadPlacement,
		XtNmuseumMode, &museumMode,
		XtNanomaly, &anomaly,
		XtNanomalySq, &anomalySq,
		XtNrightToLeftAdd, &rightToLeftAdd,
		XtNrightToLeftMult, &rightToLeftMult,
		XtNslot, &slot,
		XtNsubbase, &subbase,
		XtNbase, &base,
		XtNdisplayBase, &displayBase,
		XtNteach, &teach,
		XtNsound, &sound,
#endif
		XtNdemo, &demo,
		XtNpixmapSize, &pixmapSize, NULL);
#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			RESIZE_XPM(pixmapSize), &pixmap, NULL,
			&xpmAttributes);
	}
	if (pixmap == (Pixmap) NULL)
#endif
		pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			DEFINE_XBM);
	XtVaSetValues(topLevel,
#ifdef HAVE_MOTIF
		XmNkeyboardFocusPolicy, XmPOINTER, /* not XmEXPLICIT */
#else
		XtNinput, True,
#endif
		XtNiconPixmap, pixmap, NULL);
	XtAddCallback(abacus, XtNselectCallback,
		(XtCallbackProc) abacusCallback, (XtPointer) NULL);
#ifdef HAVE_MOTIF
	strncpy(descriptionHelp, description1Help, DESCRIPTION1_SIZE);
	strncat(descriptionHelp, "\n", 2);
	strncat(descriptionHelp, description2Help, DESCRIPTION2_SIZE);
	descriptionDialog = createHelp(menuBar, (char *) "Description",
		(char *) descriptionHelp);
	featuresDialog = createScrollHelp(menuBar, (char *) "Features",
		(char *) featuresHelp, pixmap);
	synopsisDialog = createHelp(menuBar, (char *) "Synopsis",
		(char *) synopsisHelp);
	strncpy(optionsHelp, options1Help, OPTIONS1_SIZE);
	strncat(optionsHelp, "\n", 2);
	strncat(optionsHelp, options2Help, OPTIONS2_SIZE);
	strncat(optionsHelp, "\n", 2);
	strncat(optionsHelp, options3Help, OPTIONS3_SIZE);
	strncat(optionsHelp, "\n", 2);
	strncat(optionsHelp, options4Help, OPTIONS4_SIZE);
	optionsDialog = createScrollHelp(menuBar, (char *) "Options",
		(char *) optionsHelp, pixmap);
	referencesDialog = createHelp(menuBar, (char *) "References",
		(char *) referencesHelp);
	aboutDialog = createHelp(menuBar, (char *) "About",
		(char *) aboutHelp);
	baseDialog = createBaseChangers(topLevel, (char *) "Base Settings",
		baseSpin);
	XtVaGetValues(abacus,
		XtNdelay, &i, NULL);
	delayDialog = createDelayChanger(topLevel, (char *) "Delay",
		spin);
#ifdef USE_SPIN
	createSpinner(controlRowCol, &railSlider,
		(char *) railLabel, rails, MIN_RAILS, MAX_RAILS, 2,
		(XtCallbackProc) railChangeCallback);
#else
	createSlider(controlRowCol, &railSlider,
		(char *) railLabel, rails, MIN_RAILS, MAX_RAILS, 2,
		(MAX_RAILS - MIN_RAILS + 1) * 6,
		(XtCallbackProc) railChangeCallback);
#endif
	if (mode < 0 || mode > MAX_FORMATS)
		mode = GENERIC;
	if (demo && mode == GENERIC)
		mode = CHINESE;
	formatString = XmStringCreateSimple((char *) " Format:");
	formatSubMenu = XmCreatePulldownMenu(controlRowCol,
		(char *) "formatSubMenu", NULL, 0);
	for (i = 0; i < MAX_MODES; i++) {
		formatStrings[i] =
			XmStringCreateSimple((char *) formatChoices[i]);
		XtSetArg(args[0], XmNlabelString, formatStrings[i]);
		if (i == ROMAN) {
			XtSetArg(args[1], XmNmnemonic, 'H'); /* Hand */
		} else {
			XtSetArg(args[1], XmNmnemonic, formatChoices[i][0]);
		}
		formatOptions[i] = XmCreatePushButtonGadget(formatSubMenu,
			(char *) formatChoices[i], args, 2);
		if (i == GENERIC)
			XtSetSensitive(formatOptions[i], False);
		XtAddCallback(formatOptions[i],
			XmNactivateCallback, formatCallback,
			(char *) ((size_t) i));
	}
	XtManageChildren(formatOptions, MAX_MODES);
	XtSetArg(args[0], XmNlabelString, formatString);
	XtSetArg(args[1], XmNmnemonic, 'F');
	XtSetArg(args[2], XmNsubMenuId, formatSubMenu);
	XtSetArg(args[3], XmNmenuHistory, formatOptions[mode]);
	formatMenu = XmCreateOptionMenu(controlRowCol, (char *) "FormatMenu",
		args, 4);
	for (i = 0; i < MAX_MODES; i++)
		XmStringFree(formatStrings[i]);
	XmStringFree(formatString);
	XtManageChild(formatMenu);
	piece = (topPiece == 0) ? bottomPiece : (topPiece * bottomPiece);
	piecePercent = (topPiecePercent == 0) ? bottomPiecePercent :
		(topPiecePercent * bottomPiecePercent);
	updateRadio(romanNumeralsMenu, romanNumeralsMode,
		ROMAN_NUMERALS_FORMATS, NULL);
	updateRadio(romanMarkersMenu, romanMarkersMode,
		ROMAN_MARKERS_FORMATS, NULL);
	updateToggle(displayFormatMenu, &latinToggle,
		latin, 2, (XtCallbackProc) latinCallback);
	updateToggle(displayFormatMenu, &groupToggle,
		group, 3, (XtCallbackProc) groupCallback);
	updateToggle(displayFormatMenu, &placeOnRailToggle,
		placeOnRail, 4, (XtCallbackProc) placeOnRailCallback);
	updateToggle(displayFormatMenu, &verticalToggle,
		vertical, 5, (XtCallbackProc) verticalCallback);
	updateToggle(displayFormatMenu, &stippleFrameToggle,
		stippleFrame, 6, (XtCallbackProc) stippleFrameCallback);
	updateToggle(specialRailsMenu, &signToggle, sign, 0,
		(XtCallbackProc) signCallback);
	updateRadio(pieceRailMenu, piece >> 2,
		PIECE_OPTIONS, NULL);
	updateRadio(piecePercentRailMenu, piecePercent >> 2,
		PIECE_OPTIONS, NULL);
	updateRadio(subdeckRailMenu, subbase >> 2,
		PIECE_OPTIONS, NULL);
	updateToggle(secondaryRailsMenu, &subdecksSeparatedToggle,
		subdecksSeparated, 1,
		(XtCallbackProc) subdecksSeparatedCallback);
	updateToggle(secondaryRailsMenu, &altSubbeadPlacementToggle,
		altSubbeadPlacement, 2,
		(XtCallbackProc) altSubbeadPlacementCallback);
	updateRadio(museumMenu, museumMode,
		MAX_MUSEUMS, NULL);
#ifdef ANOMALY
	anomalyMode = ANOMALY_CHOICE(anomaly, anomalySq);
	updateRadio(anomalyMenu, anomalyMode,
		ANOMALY_OPTIONS, NULL);
#endif
	updateToggle(teachMenu, &rightToLeftAddToggle,
		rightToLeftAdd, 0, (XtCallbackProc) rightToLeftAddCallback);
	updateToggle(teachMenu, &rightToLeftMultToggle,
		rightToLeftMult, 1, (XtCallbackProc) rightToLeftMultCallback);
	updateToggle(effectsMenu, &soundToggle,
		sound, 0, (XtCallbackProc) soundCallback);
	updateToggle(learnMenu, &demoToggle,
		demo, 0, (XtCallbackProc) nullCallback);
	updateToggle(learnMenu, &teachToggle,
		teach, 1, (XtCallbackProc) nullCallback);
	checkEnabled(abacus);
	XtVaSetValues(romanNumeralsMenu,
		XmNmenuHistory, romanFormatChoices[romanNumeralsMode], NULL);
	XtVaSetValues(pieceRailMenu,
		XmNmenuHistory, piece >> 2, NULL);
	XtVaSetValues(piecePercentRailMenu,
		XmNmenuHistory, piecePercent >> 2, NULL);
	XtVaSetValues(subdeckRailMenu,
		XmNmenuHistory, subbase >> 2, NULL);
	XtVaSetValues(museumMenu,
		XmNmenuHistory, museumChoices[museumMode], NULL);
#endif
	initialize(abacus);
	if (demo) {
		forceDemoParams(abacus);
		createDemo();
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	} else if (teach) {
		forceTeachParams(abacus);
#ifdef HAVE_MOTIF
		createTeach(abacus);
#elif defined(HAVE_ATHENA)
		createDialog(topLevel, teachDialogShell);
		initializeText((AbacusWidget) abacus);
#endif
#endif
	}
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(abacus), (unsigned int) AnyButton, AnyModifier,
		XtWindow(abacus), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(abacus),
		XCreateFontCursor(XtDisplay(abacus), XC_hand2));
	if (demo) {
		realizeDemo();
	}
#ifdef HAVE_EDITRES
	XtAddEventHandler(topLevel, (EventMask) 0, TRUE,
		(XtEventHandler) _XEditResCheckMessages, NULL);
#endif
	XtAppMainLoop(appCon);

#ifdef __VMS
	return 1;
#else
	return 0;
#endif
}
#endif
